package secnotify

import (
	"context"
	"encoding/json"
	"fmt"
	"math/rand"
	"strings"
	"time"

	"a.yandex-team.ru/security/impulse/models"
	"a.yandex-team.ru/security/libs/go/simplelog"
)

const YCloudSastReport = "ycloud_sast_report"

type (
	GenericVulnsTemplateParameters struct {
		AnalyzersStats []map[string]string `json:"analyzers_stats"`
		ProjectName    string              `json:"project_name"`
		ReportURL      string              `json:"report_url"`
	}
)

func filterStringsByPrefix(in []string, pattern string) (out []string) {
	for _, s := range in {
		if !strings.HasPrefix(s, pattern) {
			out = append(out, s)
		}
	}
	return
}

func (c *Controller) buildVulnerabilityReport(ctx context.Context, queue string, report *models.WebhookReport) ([]byte, error) {
	project, err := c.projectUsecase.GetByID(ctx, report.ProjectID)
	if err != nil {
		return nil, err
	}

	stats := make([]map[string]string, 0)
	for _, scanInstance := range report.Instances {
		filterAndArray := []interface{}{
			map[string]interface{}{
				"eq": map[string]interface{}{
					"v.status": models.NotReviewed,
				},
			},
			map[string]interface{}{
				"gt": map[string]interface{}{
					"v.first_found_at": time.Now().Format("2006-01-02"),
				},
			},
		}
		filterMap := map[string]interface{}{
			"and": filterAndArray,
		}

		info, err := c.scanInstanceUsecase.GetInfoByID(ctx, scanInstance.ID)
		if err != nil {
			return nil, err
		}

		vulnerabilities, err := c.vulnerabilityUsecase.FetchByScanInstances(ctx, []*models.ScanInstance{scanInstance}, info.ScanInstanceStatistics.TotalVulnerabilitiesCount, 0, filterMap)
		if err != nil {
			return nil, err
		}
		if len(vulnerabilities) == 0 {
			return nil, nil
		}

		stat := map[string]string{}
		var medium, critical, blocker int

		for _, vuln := range vulnerabilities {
			switch *vuln.Severity {
			case models.Medium:
				medium++
			case models.Critical:
				critical++
			case models.Blocker:
				blocker++
			}
		}

		// add stats on nonempty condition else continue
		if medium <= 0 && critical <= 0 && blocker <= 0 {
			continue
		}

		stat["analyzer"], err = c.scanTypeUsecase.GetScanTypeDisplayName(ctx, scanInstance.ScanTypeName)
		if err != nil {
			return nil, err
		}

		stat["count_medium"] = fmt.Sprintf("%d", medium)
		stat["count_critical"] = fmt.Sprintf("%d", critical)
		stat["count_blocker"] = fmt.Sprintf("%d", blocker)
		stats = append(stats, stat)
	}
	if len(stats) == 0 {
		return nil, nil
	}

	// need for: ban empty ABCServiceID at UI and creation level
	var assignee string
	var followers []string
	if project.ABCServiceID != 0 {
		simplelog.Info("Pick from active duty set")
		followers, err = c.ABC.GetDutys(project.ABCServiceID, true)
		if len(followers) == 0 {
			simplelog.Info("Pick from whole duty scope")
			followers, err = c.ABC.GetDutys(project.ABCServiceID, false)
			if len(followers) == 0 {
				simplelog.Info("Drop pick from dev set")
				followers, err = c.ABC.GetMembers(project.ABCServiceID, false, "development")
			}
			followers = filterStringsByPrefix(followers, "robot-")
		}
		if err != nil || len(followers) == 0 {
			return nil, err
		}
		assignee = followers[rand.Intn(len(followers))]
	} else {
		followers = []string{"gpwn"}
		assignee = "gpwn"
	}
	//////////////////////////
	//simplelog.Info("Followers: ")
	//for _, login := range followers {
	//	simplelog.Info("login: "+ login)
	//}
	//simplelog.Info("Assignee: "+ assignee)
	//////////////////////////

	params := IssueParameters{
		Queue:     queue,
		Summary:   fmt.Sprintf("[imPulse] SAST report for %s, %s", project.Name, time.Now().Format("02.01.2006")),
		Followers: followers,
		Assignee:  assignee,
		Security:  "Yes",
		Tags:      []string{"impulse", "iss_cloud"},
	}

	reportURL := fmt.Sprintf("https://impulse.sec.yandex-team.ru/organizations/%d/projects/%d?taskId=%s",
		report.OrganizationID, report.ProjectID, report.Instances[0].TaskID)

	template := &Template{
		TemplateID:      YCloudSastReport,
		IssueParameters: params,
		Parameters: GenericVulnsTemplateParameters{
			AnalyzersStats: stats,
			ProjectName:    project.Name,
			ReportURL:      reportURL,
		},
	}

	data, _ := json.Marshal(template)

	return data, nil
}
