package server

import (
	"context"
	"fmt"
	"net/http"

	"github.com/labstack/echo/v4"
	"github.com/labstack/echo/v4/middleware"

	"a.yandex-team.ru/library/go/core/buildinfo"
	"a.yandex-team.ru/library/go/core/xerrors"
	"a.yandex-team.ru/security/impulse/api/internal/middlewares/tvm"
	"a.yandex-team.ru/security/impulse/api/internal/utils"
	"a.yandex-team.ru/security/impulse/api/webhook/internal/infra"
	"a.yandex-team.ru/security/libs/go/simplelog"
)

type (
	Server struct {
		echo        *echo.Echo
		infra       *infra.Infra
		controllers serverControllers
	}
)

func New(infra *infra.Infra) (*Server, error) {
	return &Server{
		echo:        echo.New(),
		infra:       infra,
		controllers: newControllers(infra),
	}, nil
}

func (s *Server) onStart() (err error) {
	return s.infra.Start()
}

func (s *Server) onEnd() (err error) {
	return s.infra.Done()
}
func (s *Server) Prepare() error {
	err := s.onStart()
	if err != nil {
		return xerrors.Errorf("failed to start server: %w", err)
	}

	defer func() {
		err := s.onEnd()
		if err != nil {
			simplelog.Error("failed to stop", "err", err)
		}
	}()

	s.echo.HTTPErrorHandler = utils.HTTPErrorHandler

	s.echo.Pre(middleware.RemoveTrailingSlash())

	s.echo.GET("/ping", func(c echo.Context) error {
		return c.String(http.StatusOK, "pong")
	})

	s.echo.GET("/version", func(c echo.Context) error {
		return c.String(http.StatusOK, buildinfo.Info.ProgramVersion)
	})

	api := s.echo.Group("/api/v1")
	if s.infra.CFG.UseAuth {
		if err = s.controllers.Secnotify.BuildRoute(api.Group("/webhook", tvm.WithTVM(s.infra.TVM, s.infra.CFG.ImpulseTvmID))); err != nil {
			return err
		}
		if err = s.controllers.Insights.BuildRoute(api.Group("/webhook", tvm.WithTVM(s.infra.TVM, s.infra.CFG.ImpulseTvmID))); err != nil {
			return err
		}
	} else {
		if err = s.controllers.Secnotify.BuildRoute(api.Group("/webhook")); err != nil {
			return err
		}
		if err = s.controllers.Insights.BuildRoute(api.Group("/webhook")); err != nil {
			return err
		}
	}
	return nil
}

func (s *Server) ListenAndServe(port uint32) error {
	if err := s.Prepare(); err != nil {
		return err
	}
	return s.echo.Start(fmt.Sprintf(":%d", port))
}

func (s *Server) Shutdown(ctx context.Context) error {
	return s.echo.Shutdown(ctx)
}
