package main

import (
	"context"
	"os"
	"os/signal"
	"strconv"
	"syscall"
	"time"

	"a.yandex-team.ru/security/impulse/api/webhook/internal/config"
	"a.yandex-team.ru/security/impulse/api/webhook/internal/infra"
	"a.yandex-team.ru/security/impulse/api/webhook/internal/server"
	"a.yandex-team.ru/security/libs/go/simplelog"
)

const (
	shutdownDeadline = time.Second * 30
)

func newConfig() config.Config {
	var cfg config.Config
	httpPort, err := strconv.Atoi(os.Getenv("HTTP_PORT"))
	if err != nil {
		httpPort = 80
	}
	switch os.Getenv("IMPULSE_ENV") {
	case "dev":
		simplelog.SetLevel(simplelog.DebugLevel)
		cfg = config.Config{
			DBConn:           os.Getenv("DB_DSN"),
			DBPassword:       os.Getenv("DB_PASSWORD"),
			DBRetries:        3,
			SecnotifyToken:   os.Getenv("SECNOTIFY_TOKEN"),
			ABCOAuthToken:    os.Getenv("ABC_OAUTH_TOKEN"),
			ABCEndpoint:      os.Getenv("ABC_OAUTH_TOKEN"),
			Debug:            true,
			UseAuth:          false,
			UpstreamURL:      "http://localhost",
			HTTPPort:         uint32(httpPort),
			InsightsURL:      "http://project-insights-api-testing.in.yandex-team.ru",
			InsightsDstTVMID: 2031653,
		}
	case "test":
		simplelog.SetLevel(simplelog.DebugLevel)
		cfg = config.Config{
			DBConn:           os.Getenv("DB_DSN"),
			DBPassword:       os.Getenv("DB_PASSWORD"),
			DBRetries:        3,
			SecnotifyToken:   os.Getenv("SECNOTIFY_TOKEN"),
			ABCOAuthToken:    os.Getenv("ABC_OAUTH_TOKEN"),
			ABCEndpoint:      "https://abc-back.yandex-team.ru",
			Debug:            true,
			UseAuth:          true,
			ImpulseTvmID:     2016357,
			UpstreamURL:      "https://impulse-test.sec.yandex-team.ru",
			HTTPPort:         uint32(httpPort),
			InsightsURL:      "http://project-insights-api-testing.in.yandex-team.ru",
			InsightsDstTVMID: 2031653,
		}
	case "prod":
		fallthrough
	default:
		cfg = config.Config{
			DBConn:           os.Getenv("DB_DSN"),
			DBPassword:       os.Getenv("DB_PASSWORD"),
			DBRetries:        3,
			SecnotifyToken:   os.Getenv("SECNOTIFY_TOKEN"),
			ABCOAuthToken:    os.Getenv("ABC_OAUTH_TOKEN"),
			ABCEndpoint:      "https://abc-back.yandex-team.ru",
			Debug:            false,
			UseAuth:          true,
			ImpulseTvmID:     2016355,
			UpstreamURL:      "https://impulse.sec.yandex-team.ru",
			HTTPPort:         uint32(httpPort),
			InsightsURL:      "http://project-insights-api-production.yandex.net",
			InsightsDstTVMID: 2031653,
		}
	}

	return cfg
}

func main() {
	cfg := newConfig()
	srv, err := server.New(infra.New(cfg))
	if err != nil {
		panic(err)
	}

	// Setup quit channel
	quit := make(chan os.Signal)

	// Start server
	go func() {
		if err := srv.ListenAndServe(cfg.HTTPPort); err != nil {
			simplelog.Error("server server stopped", "err", err)
			// send signal manually, server failed to start
			signal.Stop(quit)
			quit <- syscall.SIGTERM
		}
	}()

	// Wait termination
	signal.Notify(quit, syscall.SIGINT, syscall.SIGTERM)
	<-quit

	ctx, cancel := context.WithTimeout(context.Background(), shutdownDeadline)
	defer cancel()

	if err := srv.Shutdown(ctx); err != nil {
		simplelog.Error("failed to shutdown server", "err", err)
	}
}
