package config

import (
	"fmt"
	"strings"
	"time"

	"a.yandex-team.ru/library/go/yandex/tvm"
	"a.yandex-team.ru/security/impulse/api/internal/db"
	"a.yandex-team.ru/security/impulse/api/internal/sandbox"
	"a.yandex-team.ru/security/impulse/api/internal/secnotify"
	"a.yandex-team.ru/security/impulse/pkg/queue"
	"a.yandex-team.ru/security/impulse/pkg/s3"
)

type (
	Config struct {
		DBConn                    string
		DBPassword                string
		DBRetries                 int
		SqsEndpoint               string
		SqsAccount                string
		SqsRetries                int
		SqsOAuthToken             string
		Concurrency               int
		SandboxEndpoint           string
		SandboxOAuthToken         string
		SandboxRetryTimeout       int
		CodeQLBuildTaskInterval   int64
		CodeQLBuildTaskBatchSize  int
		Debug                     bool
		SecNofityURL              string
		SecNotifyDstTvmID         tvm.ClientID
		SecNofityToken            string
		UseAuth                   bool
		SendNotificationsInterval int
		S3Endpoint                string
		S3Region                  string
		S3AccessKeyID             string
		S3SecretAccessKey         string
	}
)

func (c *Config) QueueConfig() *queue.Options {
	return &queue.Options{
		Endpoint:   c.SqsEndpoint,
		Account:    c.SqsAccount,
		MaxRetries: c.SqsRetries,
		AuthType:   queue.AuthOAuth,
		OAuthToken: c.SqsOAuthToken,
	}
}

func (c *Config) SecNotifyConfig() *secnotify.Options {
	return &secnotify.Options{
		Endpoint: c.SecNofityURL,
		Token:    c.SecNofityToken,
		DstTvmID: c.SecNotifyDstTvmID,
	}
}

func (c *Config) DBConfig() *db.Options {
	return &db.Options{
		DSN:             c.DBConn,
		Password:        c.DBPassword,
		QueryRetries:    c.DBRetries,
		MaxOpenConns:    10,
		MaxIdleConns:    2,
		ConnMaxLifetime: time.Hour,
	}
}

func (c *Config) SandboxConfig() *sandbox.Options {
	return &sandbox.Options{
		Endpoint:   c.SandboxEndpoint,
		OAuthToken: c.SandboxOAuthToken,
	}
}

func (c *Config) QueueURL(queueName string) string {
	return fmt.Sprintf("%s/%s/%s",
		strings.TrimRight(c.SqsEndpoint, "/"),
		c.SqsAccount,
		queueName,
	)
}

func (c *Config) TasksQueueURL() string {
	return fmt.Sprintf("%s/%s/%s",
		strings.TrimRight(c.SqsEndpoint, "/"),
		c.SqsAccount,
		"tasks",
	)
}

func (c *Config) ReportsQueueURL() string {
	return fmt.Sprintf("%s/%s/%s",
		strings.TrimRight(c.SqsEndpoint, "/"),
		c.SqsAccount,
		"reports",
	)
}

func (c *Config) S3Config() *s3.Options {
	return &s3.Options{
		Endpoint:        c.S3Endpoint,
		Region:          c.S3Region,
		AccessKeyID:     c.S3AccessKeyID,
		SecretAccessKey: c.S3SecretAccessKey,
	}
}
