package common

import (
	"encoding/json"
	"regexp"

	"a.yandex-team.ru/security/impulse/models"
)

type (
	ReportResult struct {
		models.ResultMeta
		Report []*models.NewVulnerabilityDeduplicationRequestDTO `json:"report"`
	}

	DedupImpl interface {
		UnmarshalReport(reportRaw []byte) (*ReportResult, error)
		Filter(vulns []*models.NewVulnerabilityDeduplicationRequestDTO, includeFilenamePatterns []string,
			excludeFilenamePatterns []string) ([]*models.NewVulnerabilityDeduplicationRequestDTO, error)
		Deduplicate(oldVulns []*models.VulnerabilityDeduplicationResponseDTO,
			vulns []*models.NewVulnerabilityDeduplicationRequestDTO) ([]*models.NewVulnerabilityDeduplicationRequestDTO,
			[]*models.DeduplicatedVulnerabilityDeduplicationRequestDTO, error)
	}
)

func UnmarshalReport(reportRaw []byte) (*ReportResult, error) {
	report := new(ReportResult)
	err := json.Unmarshal(reportRaw, &report)
	if err != nil {
		return nil, err
	}
	return report, nil
}

func Filter(keyPropertyName string, vulns []*models.NewVulnerabilityDeduplicationRequestDTO, includeFilenamePatterns []string,
	excludeFilenamePatterns []string) ([]*models.NewVulnerabilityDeduplicationRequestDTO, error) {

	filtredIncludedVulns := make([]*models.NewVulnerabilityDeduplicationRequestDTO, 0)
	for _, vuln := range vulns {
		allowed := len(includeFilenamePatterns) == 0 // allow all vulns if IncludeFilenamePatterns is empty
		for _, includeFilenamePattern := range includeFilenamePatterns {
			match, err := regexp.MatchString(includeFilenamePattern, vuln.KeyProperties[keyPropertyName].(string))
			if err != nil {
				return nil, err
			}
			if match {
				allowed = true
			}
		}
		if allowed {
			filtredIncludedVulns = append(filtredIncludedVulns, vuln)
		}
	}

	filtredVulns := make([]*models.NewVulnerabilityDeduplicationRequestDTO, 0)
	for _, vuln := range filtredIncludedVulns {
		allowed := true
		for _, excludeFilenamePattern := range excludeFilenamePatterns {
			match, err := regexp.MatchString(excludeFilenamePattern, vuln.KeyProperties[keyPropertyName].(string))
			if err != nil {
				return nil, err
			}
			if match {
				allowed = false
			}
		}
		if allowed {
			filtredVulns = append(filtredVulns, vuln)
		}
	}

	return filtredVulns, nil
}
