package worker

import (
	"context"
	"sync"
	"time"

	"a.yandex-team.ru/security/impulse/api/internal/sandbox"
	"a.yandex-team.ru/security/impulse/models"
	"a.yandex-team.ru/security/libs/go/simplelog"
)

func (w *Worker) codeQLProcessor(wg *sync.WaitGroup) {
	wg.Add(1)
	defer wg.Done()

	loc, _ := time.LoadLocation("Europe/Moscow")
	prevTime := time.Now().In(loc)
	currentAmount := 0

	for {
		currentTime := time.Now().In(loc)
		if currentTime.Unix()-prevTime.Unix() > w.CFG.CodeQLBuildTaskInterval {
			prevTime = currentTime
			currentAmount = 0
		}
		if currentAmount >= w.CFG.CodeQLBuildTaskBatchSize {
			time.Sleep(60 * time.Second)
			continue
		}

		buildTasks, err := w.codeQLUsecase.GetOldestBuildTasks(w.ctx,
			currentTime.Unix()-w.CFG.CodeQLBuildTaskInterval,
			w.CFG.CodeQLBuildTaskBatchSize-currentAmount,
		)
		if err == context.Canceled {
			break
		}
		if err != nil {
			simplelog.Error("GetOldestBuildTasks error", "err", err)
			continue
		}
		if len(buildTasks) == 0 {
			simplelog.Info("GetOldestBuildTasks empty result, sleeping..")
			time.Sleep(60 * time.Second)
			continue
		}

		for _, buildTask := range buildTasks {
			latestTasks, err := w.taskUsecase.List(w.ctx, buildTask.OrganizationID, buildTask.ProjectID, 1, 0)
			if err == context.Canceled {
				break
			}
			if err != nil || len(latestTasks) == 0 {
				continue
			}
			normalizedParameters, err := w.taskUsecase.NormalizeTaskParameters(latestTasks[0].Parameters)
			if err != nil {
				simplelog.Error("Task parameters normalization error", "err", err)
				continue
			}
			repositories, ok := normalizedParameters["repositories"].([]models.Repository)
			if !ok || len(repositories) == 0 {
				simplelog.Error("Task parameters no repositories field or empty")
				continue
			}
			repositoryURL := repositories[0].URL

			sandboxTaskID, err := w.Sandbox.CreateBuildCodeQLTask(buildTask.OrganizationID,
				buildTask.ProjectID,
				repositoryURL,
			)
			if err != nil {
				simplelog.Error("CreateBuildCodeQLTask error", "err", err)
				continue
			}
			err = w.Sandbox.StartTask(sandboxTaskID)
			if err != nil {
				simplelog.Error("StartTask error", "sandboxTaskID", sandboxTaskID, "err", err)
				switch err.(type) {
				case sandbox.TooManyRequests:
					time.Sleep(60 * time.Second)
				}
				continue
			}
			simplelog.Info("StartTask ok", "sandboxTaskID", sandboxTaskID)

			currentAmount += 1

			_, err = w.codeQLUsecase.CreateBuildTask(w.ctx,
				&models.CodeQLBuildTask{
					OrganizationID: buildTask.OrganizationID,
					ProjectID:      buildTask.ProjectID,
					SandboxTaskID:  sandboxTaskID,
					StartTime:      time.Now().In(loc).Unix(),
				})
			if err != nil {
				simplelog.Error("CreateBuildTask error", "err", err)
				continue
			}
		}

		time.Sleep(5 * time.Second)
	}
}
