package worker

import (
	"context"
	"sync"
	"time"

	"github.com/aws/aws-sdk-go/service/sqs"

	"a.yandex-team.ru/library/go/core/xerrors"
	_codeQLRepo "a.yandex-team.ru/security/impulse/api/repositories/codeqldb"
	_cronRepo "a.yandex-team.ru/security/impulse/api/repositories/cron"
	_projectRepo "a.yandex-team.ru/security/impulse/api/repositories/project"
	_scanRepo "a.yandex-team.ru/security/impulse/api/repositories/scan"
	_scanInstanceRepo "a.yandex-team.ru/security/impulse/api/repositories/scaninstance"
	_scanTypeRepo "a.yandex-team.ru/security/impulse/api/repositories/scantype"
	_taskRepo "a.yandex-team.ru/security/impulse/api/repositories/task"
	_vulnerabilityRepo "a.yandex-team.ru/security/impulse/api/repositories/vulnerability"
	_vulnerabilityCategoryRepo "a.yandex-team.ru/security/impulse/api/repositories/vulnerabilitycategory"
	_vulnerabilityTotalStatisticsRepo "a.yandex-team.ru/security/impulse/api/repositories/vulnerabilitytotalstatistics"
	_workflowRepo "a.yandex-team.ru/security/impulse/api/repositories/workflow"
	_codeQLUsecase "a.yandex-team.ru/security/impulse/api/usecases/codeqldb"
	_cronUsecase "a.yandex-team.ru/security/impulse/api/usecases/cron"
	_projectUsecase "a.yandex-team.ru/security/impulse/api/usecases/project"
	_scanUsecase "a.yandex-team.ru/security/impulse/api/usecases/scan"
	_scanInstanceUsecase "a.yandex-team.ru/security/impulse/api/usecases/scaninstance"
	_taskUsecase "a.yandex-team.ru/security/impulse/api/usecases/task"
	_vulnerabilityUsecase "a.yandex-team.ru/security/impulse/api/usecases/vulnerability"
	_vulnerabilityCategoryUsecase "a.yandex-team.ru/security/impulse/api/usecases/vulnerabilitycategory"
	_vulnerabilityTotalStatisticsUsecase "a.yandex-team.ru/security/impulse/api/usecases/vulnerabilitytotalstatistics"
	_infra "a.yandex-team.ru/security/impulse/api/worker/internal/infra"
	_queue "a.yandex-team.ru/security/impulse/pkg/queue"
	"a.yandex-team.ru/security/libs/go/simplelog"
)

const (
	backoffTimeout = time.Second * 10
)

type (
	Worker struct {
		ctx          context.Context
		exitChan     chan struct{}
		shutDownFunc context.CancelFunc
		_infra.Infra

		taskUsecase                         _taskUsecase.Usecase
		projectUsecase                      _projectUsecase.Usecase
		codeQLUsecase                       _codeQLUsecase.Usecase
		cronUsecase                         _cronUsecase.Usecase
		vulnerabilityUsecase                _vulnerabilityUsecase.Usecase
		scanInstanceUsecase                 _scanInstanceUsecase.Usecase
		scanUsecase                         _scanUsecase.Usecase
		vulnerabilityCategoryUsecase        _vulnerabilityCategoryUsecase.Usecase
		vulnerabilityTotalStatisticsUsecase _vulnerabilityTotalStatisticsUsecase.Usecase
	}
)

func New(infra _infra.Infra) *Worker {
	ctx, shutdown := context.WithCancel(context.Background())

	return &Worker{
		ctx:          ctx,
		shutDownFunc: shutdown,
		exitChan:     make(chan struct{}),
		Infra:        infra,
	}
}

func (w *Worker) onStart() (err error) {

	taskRepo := _taskRepo.NewTaskRepository(w.DB)
	vulnerabilityRepo := _vulnerabilityRepo.NewVulnerabilityRepository(w.DB)
	vulnerabilityCategoryRepo := _vulnerabilityCategoryRepo.NewVulnerabilityCategoryRepository(w.DB)
	vulnerabilityTotalStatisticsRepo := _vulnerabilityTotalStatisticsRepo.NewVulnerabilityTotalStatisticsRepository(w.DB)
	projectRepo := _projectRepo.NewProjectRepository(w.DB)
	scanRepository := _scanRepo.NewScanRepository(w.DB)
	scanInstanceRepo := _scanInstanceRepo.NewScanInstanceRepository(w.DB)
	codeQLRepo := _codeQLRepo.NewCodeQLDatabaseRepository(w.DB)
	cronRepo := _cronRepo.NewCronRepository(w.DB)
	scanTypeRepo := _scanTypeRepo.NewScanTypeRepository(w.DB)
	workflowRepo := _workflowRepo.NewWorkflowRepository(w.DB)

	w.scanInstanceUsecase = _scanInstanceUsecase.NewScanInstanceStatusUsecase(scanRepository, scanInstanceRepo)
	w.codeQLUsecase = _codeQLUsecase.NewCodeQLDatabaseUsecase(codeQLRepo)
	w.projectUsecase = _projectUsecase.NewProjectUsecase(projectRepo, scanInstanceRepo, w.scanInstanceUsecase)
	w.scanUsecase = _scanUsecase.NewScanUsecase(scanRepository, projectRepo, scanTypeRepo, scanInstanceRepo, workflowRepo)
	w.cronUsecase = _cronUsecase.NewCronUsecase(cronRepo)
	w.vulnerabilityTotalStatisticsUsecase = _vulnerabilityTotalStatisticsUsecase.NewVulnerabilityTotalStatisticsUsecase(
		vulnerabilityTotalStatisticsRepo)
	w.vulnerabilityUsecase = _vulnerabilityUsecase.NewVulnerabilityUsecase(vulnerabilityRepo,
		w.vulnerabilityTotalStatisticsUsecase)
	w.vulnerabilityCategoryUsecase = _vulnerabilityCategoryUsecase.NewVulnerabilityCategoryUsecase(vulnerabilityCategoryRepo)
	w.taskUsecase = _taskUsecase.NewTaskUsecase(taskRepo)

	return
}

func (w *Worker) onEnd() (err error) {

	return
}

func (w *Worker) runProducer(opts *_queue.ReceiveOptions, queue chan<- *sqs.Message) {
	for {
		select {
		case <-w.ctx.Done():
			simplelog.Info("worker stopped")
			close(queue)
			return
		default:
		}

		messages, err := w.Queue.ReceiveMessage(opts)
		if err != nil {
			simplelog.Error("failed to receive responses", "err", err)
			time.Sleep(backoffTimeout)
			continue
		}

		for _, msg := range messages {
			simplelog.Info("recieve", "msg", msg)
			queue <- msg
		}
	}
}

func (w *Worker) Start() error {
	defer close(w.exitChan)

	err := w.onStart()
	if err != nil {
		return xerrors.Errorf("failed to start worker: %w", err)
	}

	defer func() {
		err := w.onEnd()
		if err != nil {
			simplelog.Error("failed to stop", "err", err)
		}
	}()

	simplelog.Info("worker started")

	tasksQueueURL := w.CFG.TasksQueueURL()
	taskMessagesQueue := make(chan *sqs.Message, w.CFG.Concurrency)

	go w.runProducer(&_queue.ReceiveOptions{
		QueueURL:            tasksQueueURL,
		MaxNumberOfMessages: 1,
	}, taskMessagesQueue)
	wg := sync.WaitGroup{}
	for i := 0; i < w.CFG.Concurrency; i++ {
		go w.taskProcessor(&wg, taskMessagesQueue)
	}
	go w.reportLoader(&wg)
	go w.notificationProcessor(&wg)
	go w.cronProcessor(&wg)
	go w.codeQLProcessor(&wg)

	wg.Wait()
	return nil
}

func (w *Worker) Shutdown(ctx context.Context) error {
	w.shutDownFunc()

	// grateful wait processor wg
	select {
	case <-w.exitChan:
		// completed normally
		return nil
	case <-ctx.Done():
		// timed out
		return xerrors.Errorf("time out")
	}
}
