package worker

import (
	"context"
	"encoding/json"
	"sync"
	"time"

	"github.com/gofrs/uuid"

	"a.yandex-team.ru/drive/library/go/cron"
	"a.yandex-team.ru/security/impulse/models"
	"a.yandex-team.ru/security/impulse/pkg/queue"
	"a.yandex-team.ru/security/libs/go/simplelog"
)

func (w *Worker) cronProcessor(wg *sync.WaitGroup) {
	wg.Add(1)
	defer wg.Done()

	loc, _ := time.LoadLocation("Europe/Moscow")

	for {
		currentTime := time.Now().In(loc)
		tasks, err := w.cronUsecase.ListAll(w.ctx)
		if err == context.Canceled {
			break
		}
		if err != nil {
			simplelog.Error("ListAll error", "err", err)
			time.Sleep(5 * time.Second)
			continue
		}
		for _, cronTask := range tasks {
			if currentTime.Unix() > cronTask.NextTime {
				if !cronTask.IsRunning {
					spec, _ := cron.Parse(cronTask.Spec)
					nextTime := spec.Next(currentTime).Unix()
					err = w.cronUsecase.UpdateNextTime(w.ctx, cronTask.OrganizationID, cronTask.ProjectID, cronTask.ID, nextTime)
					if err != nil {
						simplelog.Error("UpdateNextTime error", "err", err)
					}
					continue
				}

				taskID := uuid.Must(uuid.NewV4()).String()
				task := models.Task{
					TaskID:          taskID,
					OrganizationID:  cronTask.OrganizationID,
					ProjectID:       cronTask.ProjectID,
					Parameters:      cronTask.Parameters,
					Analysers:       cronTask.Analysers,
					StartTime:       time.Now().Unix(),
					Status:          models.Created,
					CronID:          cronTask.ID,
					CallbackURL:     cronTask.CallbackURL,
					NonTemplateScan: cronTask.NonTemplateScan,
				}
				err := w.taskUsecase.Create(w.ctx, &task)
				if err != nil {
					simplelog.Error("Create task error", "err", err)
					time.Sleep(5 * time.Second)
					continue
				}

				normalizedParameters, err := w.taskUsecase.NormalizeTaskParameters(cronTask.Parameters)
				if err != nil {
					simplelog.Error("Task parameters normalization error", "err", err)
					time.Sleep(5 * time.Second)
					continue
				}

				msg, _ := json.Marshal(&models.TaskMessageDTO{
					OrganizationID: cronTask.OrganizationID,
					ProjectID:      cronTask.ProjectID,
					TaskID:         taskID,
					Parameters:     normalizedParameters,
					Analysers:      cronTask.Analysers,
				})
				_, err = w.Queue.SendMessage(&queue.SendOptions{
					QueueURL: w.CFG.TasksQueueURL(),
					Msg:      string(msg),
				})
				if err != nil {
					simplelog.Error("failed to send message", "err", err)
					time.Sleep(5 * time.Second)
					continue
				}

				spec, _ := cron.Parse(cronTask.Spec)
				nextTime := spec.Next(currentTime).Unix()
				err = w.cronUsecase.UpdateNextTime(w.ctx, cronTask.OrganizationID, cronTask.ProjectID, cronTask.ID, nextTime)
				if err != nil {
					simplelog.Error("UpdateNextTime error", "err", err)
				}
			}
		}

		time.Sleep(5 * time.Second)
	}
}
