package worker

import (
	"context"
	"fmt"
	"sync"
	"time"

	_secnotify "a.yandex-team.ru/security/impulse/api/internal/secnotify"
	"a.yandex-team.ru/security/impulse/models"
	"a.yandex-team.ru/security/libs/go/simplelog"
)

const vulnerabilityURLTmpl = "https://impulse.sec.yandex-team.ru/organizations/%d/projects/%d/scans/%s/instances/%d/vulnerabilities/%d"
const robotName = "robot-c3po"

func generateTicketTitle(vulnerability *models.Vulnerability) string {
	if len(vulnerability.KeyProperties) != 0 && len(vulnerability.DisplayProperties) != 0 {
		switch vulnerability.ScanTypeName {
		case string(models.YADI):
			return fmt.Sprintf("%s in %s %s", vulnerability.CategoryName, vulnerability.KeyProperties["package_name"],
				vulnerability.KeyProperties["version"])
		case string(models.YODAX):
			return fmt.Sprintf("%s", vulnerability.DisplayProperties["summary"])
		case string(models.RIPS):
			return vulnerability.CategoryName
		default:
			return fmt.Sprintf("%s in %s", vulnerability.CategoryName, vulnerability.KeyProperties["filename"])
		}
	} else {
		return vulnerability.CategoryName
	}
}

func generateTicketBody(vulnerability *models.Vulnerability, project *models.ProjectInfo) string {
	vulnerabilityURL := fmt.Sprintf(vulnerabilityURLTmpl, project.OrganizationID,
		vulnerability.ProjectID, vulnerability.ScanTypeName, vulnerability.ScanInstanceID, vulnerability.ID)
	return fmt.Sprintf("Link to the vulnerability: %s", vulnerabilityURL)
}

func (w *Worker) notificationProcessor(wg *sync.WaitGroup) {
	wg.Add(1)
	defer wg.Done()
	ticker := time.NewTicker(time.Duration(w.CFG.SendNotificationsInterval) * time.Second)

	for range ticker.C {
		simplelog.Info("Start notification processor run")
		projects, err := w.projectUsecase.ListWithEnabledNotifications(w.ctx)

		if err == context.Canceled {
			break
		}
		if err != nil {
			simplelog.Error("List notifications error", "err", err)
			continue
		}
		if len(projects) == 0 {
			continue
		}
		for _, project := range projects {
			since := time.Now().Add(time.Duration(-1*w.CFG.SendNotificationsInterval) * time.Second)
			vulns, err := w.vulnerabilityUsecase.FetchLatestByProjectIDWithoutTicket(w.ctx, project.ID,
				since, 100, 0, project.NotificationSettings.Filter)
			if err == context.Canceled {
				break
			}
			if err != nil {
				simplelog.Error("FetchLatestWaitingNotification error", "err", err)
				continue
			}
			if len(vulns) == 0 {
				continue
			}
			if project.TrackerQueue == "" {
				simplelog.Error("No tracker queue for project", "project_id", project.ID)
			}
			simplelog.Info("Creating tickets", "amount", len(vulns), "project_id", project.ID)
			for _, vulnerability := range vulns {
				tags := []string{"impulse", vulnerability.ScanTypeName}
				issueParameters := _secnotify.CreateTicketIssueParameters{
					Queue:            project.TrackerQueue,
					Summary:          generateTicketTitle(vulnerability),
					Tags:             tags,
					CreatedBy:        robotName,
					Followers:        project.NotificationSettings.Followers,
					Assignee:         project.NotificationSettings.Assignee,
					Security:         _secnotify.Yes,
					SecuritySeverity: _secnotify.SeverityTrackerMap[*vulnerability.Severity],
				}
				trackerTicket, err := w.Secnotify.CreateTicket(w.ctx, "impulse_results",
					map[string]interface{}{"ticket_body": generateTicketBody(vulnerability, project)}, issueParameters)
				if err != nil {
					simplelog.Error("Secnotify Create Ticket error", "err", err)
				}
				err = w.vulnerabilityUsecase.UpdateTrackerTicketByID(w.ctx, trackerTicket, vulnerability.ID)
				if err != nil {
					simplelog.Error("Update vulnerability Ticket error", "err", err)

				}
			}
		}

	}
}
