package worker

import (
	"encoding/json"
	"errors"
	"fmt"
	"net/url"
	"strings"
	"sync"
	"time"

	"a.yandex-team.ru/security/impulse/api/worker/internal/dedup/blackduck"
	"a.yandex-team.ru/security/impulse/api/worker/internal/dedup/common"
	"a.yandex-team.ru/security/impulse/api/worker/internal/dedup/findsecbugs"
	"a.yandex-team.ru/security/impulse/api/worker/internal/dedup/rips"
	"a.yandex-team.ru/security/impulse/api/worker/internal/dedup/sonar"
	"a.yandex-team.ru/security/impulse/api/worker/internal/dedup/yadi"
	"a.yandex-team.ru/security/impulse/api/worker/internal/dedup/yodax"
	"a.yandex-team.ru/security/impulse/models"
	"a.yandex-team.ru/security/impulse/pkg/queue"
	"a.yandex-team.ru/security/libs/go/simplelog"
)

const maxReportSize = 400 * 1024 * 1024

func getDedupImpl(reportType models.ScanTypeName) (common.DedupImpl, error) {
	switch reportType {
	case models.CODEQL:
		fallthrough
	case models.ESLINT:
		fallthrough
	case models.GOSEC:
		fallthrough
	case models.SEMGREP:
		fallthrough
	case models.BANDIT:
		fallthrough
	case models.BANDIT2:
		fallthrough
	case models.BANDIT3:
		return sonar.NewDedupImpl(), nil
	case models.YADI:
		return yadi.NewDedupImpl(), nil
	case models.RIPS:
		return rips.NewDedupImpl(), nil
	case models.YODAX:
		return yodax.NewDedupImpl(), nil
	case models.FINDSECBUGS:
		return findsecbugs.NewDedupImpl(), nil
	case models.BLACKDUCK:
		return blackduck.NewDedupImpl(), nil
	default:
		return nil, errors.New("unsupported report type")
	}
}

func extractPatterns(parameters models.TaskParameters, paramName string) (patterns []string) {
	if tmpPatterns, ok := parameters[paramName]; !ok {
		patterns = []string{}
	} else {
		patterns, ok = tmpPatterns.([]string)
		if !ok {
			patterns = []string{}
		}
	}
	return
}

type dedupParams struct {
	dedupImpl        common.DedupImpl
	task             *models.Task
	scantype         models.ScanTypeName
	normalizedReport *common.ReportResult
}

func (w *Worker) deduplicate(params *dedupParams) error {
	includePatterns := extractPatterns(params.task.Parameters, "include_patterns")
	excludePatterns := extractPatterns(params.task.Parameters, "exclude_patterns")

	filtredVulnerabilities, err := params.dedupImpl.Filter(params.normalizedReport.Report, includePatterns, excludePatterns)
	if err != nil {
		return fmt.Errorf("failed to filter vulnerabilities: %s", err)
	}

	currentScan, err := w.scanUsecase.GetOrCreateByProjectIDAndScanTypeName(w.ctx, params.task.ProjectID, string(params.scantype))
	if err != nil {
		return fmt.Errorf("failed to get or create scan: %s", err)

	}
	currentVulns, err := w.vulnerabilityUsecase.FetchByScanID(w.ctx, currentScan.ID)
	if err != nil {
		return fmt.Errorf("failed to get vulnerabilities: %s", err)

	}
	currentVulnsResponse := []*models.VulnerabilityDeduplicationResponseDTO{}
	for _, vuln := range currentVulns {
		currentVulnsResponse = append(currentVulnsResponse, &models.VulnerabilityDeduplicationResponseDTO{ID: vuln.ID, KeyProperties: vuln.KeyProperties})
	}
	newVulnerabilitiesDeduplicationRequests, deduplicatedVulnerabilitiesDeduplicationRequests, err := params.dedupImpl.Deduplicate(currentVulnsResponse,
		filtredVulnerabilities)
	if err != nil {
		return fmt.Errorf("failed to deduplicate vulnerabilities: %s", err)
	}
	simplelog.Info("loader", "new", len(newVulnerabilitiesDeduplicationRequests), "deduplicated", len(deduplicatedVulnerabilitiesDeduplicationRequests))

	categoryMap, err := w.vulnerabilityCategoryUsecase.ListByScanType(w.ctx, currentScan.ScanTypeID)
	if err != nil {
		return fmt.Errorf("failed to list categories: %s", err)
	}

	allVulnerabilities := make([]*models.Vulnerability, 0)
	if len(newVulnerabilitiesDeduplicationRequests) > 0 {
		newVulnerabilities := make([]*models.Vulnerability, 0)
		for _, vulnerabilityDTO := range newVulnerabilitiesDeduplicationRequests {
			categoryID, ok := categoryMap[vulnerabilityDTO.Category]
			if !ok {
				categoryID, err = w.vulnerabilityCategoryUsecase.Insert(w.ctx, currentScan.ScanTypeID, vulnerabilityDTO.Category)
				if err != nil {
					return fmt.Errorf("failed to insert new vulnerability category: %s", err)
				}
				categoryMap[vulnerabilityDTO.Category] = categoryID
			}
			newVulnerabilities = append(newVulnerabilities,
				&models.Vulnerability{
					ScanID:            currentScan.ID,
					Severity:          &vulnerabilityDTO.Severity,
					CategoryID:        categoryID,
					KeyProperties:     vulnerabilityDTO.KeyProperties,
					DisplayProperties: vulnerabilityDTO.DisplayProperties,
				})
		}

		allVulnerabilities, err = w.vulnerabilityUsecase.InsertByScanIDAndLastUpdateToken(w.ctx,
			currentScan.ID, currentScan.LastUpdateToken, newVulnerabilities)
		if err != nil {
			return fmt.Errorf("failed to insert new vulnerabilities: %s", err)
		}
	}

	for _, vulnerabilityDTO := range deduplicatedVulnerabilitiesDeduplicationRequests {
		allVulnerabilities = append(allVulnerabilities,
			&models.Vulnerability{
				ID:                vulnerabilityDTO.ID,
				ScanID:            currentScan.ID,
				Severity:          &vulnerabilityDTO.Severity,
				CategoryID:        categoryMap[vulnerabilityDTO.Category],
				KeyProperties:     vulnerabilityDTO.KeyProperties,
				DisplayProperties: vulnerabilityDTO.DisplayProperties,
			})
	}
	currentScanInstance, err := w.scanInstanceUsecase.Create(w.ctx, &models.ScanInstance{
		ScanID:       currentScan.ID,
		TaskID:       params.task.TaskID,
		RawReportURL: params.normalizedReport.RawReportURL,
		CommitHash:   params.normalizedReport.CommitHash,
		StartTime:    time.Unix(params.normalizedReport.StartTime, 0),
		EndTime:      time.Unix(params.normalizedReport.EndTime, 0),
	})
	if err != nil {
		return fmt.Errorf("failed to create scaninstance: %s", err)
	}

	if len(allVulnerabilities) > 0 {
		_, err = w.scanInstanceUsecase.InsertVulnerabilities2ScanInstance(w.ctx, currentScanInstance,
			allVulnerabilities)
		if err != nil {
			return fmt.Errorf("failed to insert Vulnerabilities2ScanInstance: %s", err)
		}

		err = w.vulnerabilityUsecase.UpdateFromScanInstanceVulnerabilities(w.ctx, currentScanInstance.ID)
		if err != nil {
			return fmt.Errorf("failed to UpdateFromScanInstanceVulnerabilities: %s", err)
		}

		if err := w.vulnerabilityTotalStatisticsUsecase.UpdateStatisticsByProjectID(w.ctx, params.task.ProjectID); err != nil {
			return fmt.Errorf("UpdateStatisticsByProjectID failed: %s", err)
		}
	}
	return nil
}

func (w *Worker) processReport(body string) error {
	var loadRequest models.ReportLoadRequest
	err := json.Unmarshal([]byte(body), &loadRequest)
	if err != nil {
		return fmt.Errorf("failed to unmarshal task message: %s", err)
	}
	simplelog.Info("loader", "report", loadRequest.NormalizedReportURL)

	dedupImpl, err := getDedupImpl(loadRequest.Type)
	if err != nil {
		return err
	}

	url, err := url.Parse(loadRequest.NormalizedReportURL)
	if err != nil {
		return fmt.Errorf("failed to parse normalized URL: %s", err)
	}
	bucket := strings.Split(url.Host, ".")[0]
	key := strings.TrimLeft(url.Path, "/")

	size, err := w.S3.FileSize(bucket, key)
	if err != nil {
		return fmt.Errorf("failed to get file size: %s", err)
	}
	if size >= maxReportSize {
		return fmt.Errorf("file %s too large: %s", loadRequest.NormalizedReportURL, err)
	}
	normaizedReportBytes, err := w.S3.S3Download(bucket, key)
	if err != nil {
		return fmt.Errorf("failed to download report: %s", err)
	}
	normalizedReport, err := dedupImpl.UnmarshalReport(normaizedReportBytes)
	if err != nil {
		return fmt.Errorf("failed to unmarshal normailzed report: %s", err)
	}

	task, err := w.taskUsecase.GetByTaskID(w.ctx, normalizedReport.TaskID)
	if err != nil {
		return fmt.Errorf("failed to get task: %s", err)
	}

	err = w.deduplicate(&dedupParams{
		dedupImpl:        dedupImpl,
		task:             task,
		scantype:         loadRequest.Type,
		normalizedReport: normalizedReport,
	})
	if err != nil {
		task.Status = models.Failed
		if err := w.taskUsecase.Update(w.ctx, task); err != nil {
			return fmt.Errorf("failed to update task status: %s", err)
		}
		return err
	}

	finished, err := w.taskUsecase.CheckFinished(w.ctx, task.TaskID)
	if err != nil {
		return fmt.Errorf("failed to check finished state: %s", err)
	}
	if finished {
		loc, _ := time.LoadLocation("Europe/Moscow")
		task.EndTime = time.Now().In(loc).Unix()
		task.Status = models.Finished
		if task.CallbackURL != "" {
			instances, err := w.scanInstanceUsecase.ListByTaskID(w.ctx, task.TaskID)
			if err != nil {
				simplelog.Error("ListByTaskID", "error", err)
				task.Status = models.FinishedCallbackFailed
			} else {
				data, _ := json.Marshal(&models.WebhookReport{
					OrganizationID: task.OrganizationID,
					ProjectID:      task.ProjectID,
					Instances:      instances,
				})
				err = w.CallbackAPI.Do(w.ctx, task.CallbackURL, data)
				if err != nil {
					simplelog.Error("callback", "error", err)
					task.Status = models.FinishedCallbackFailed
				}
			}
		}
		err := w.taskUsecase.Update(w.ctx, task)
		if err != nil {
			simplelog.Error("failed to update task status", "err", err)
		}
		simplelog.Info("Task finished", "task_id", task.TaskID)
	}
	return nil
}

func (w *Worker) reportLoader(wg *sync.WaitGroup) {
	wg.Add(1)
	defer wg.Done()
	for {
		select {
		case <-w.ctx.Done():
			return
		default:
		}

		messages, err := w.Queue.ReceiveMessage(&queue.ReceiveOptions{
			QueueURL:            w.CFG.ReportsQueueURL(),
			MaxNumberOfMessages: 1,
		})
		if err != nil {
			simplelog.Error("failed to receive report message", "err", err)
			time.Sleep(backoffTimeout)
			continue
		}
		for _, msg := range messages {
			if err := w.processReport(*msg.Body); err != nil {
				simplelog.Error("process report failed", "err", err)
			}
			if err := w.Queue.DeleteMessage(&queue.DeleteOptions{
				QueueURL:      w.CFG.ReportsQueueURL(),
				ReceiptHandle: msg.ReceiptHandle,
			}); err != nil {
				simplelog.Error("failed to delete message from queue", "err", err)
			}
		}
	}
}
