package worker

import (
	"encoding/json"
	"strings"
	"sync"
	"time"

	"github.com/aws/aws-sdk-go/service/sqs"
	"github.com/gofrs/uuid"

	"a.yandex-team.ru/security/impulse/api/internal/sandbox"
	"a.yandex-team.ru/security/impulse/models"
	"a.yandex-team.ru/security/impulse/pkg/queue"
	"a.yandex-team.ru/security/libs/go/simplelog"
)

func (w *Worker) startSandboxTask(taskMessage *models.TaskMessageDTO) (int, error) {
	sandboxTaskID, err := w.Sandbox.CreateTask(taskMessage, true)
	if err != nil {
		return 0, err
	}

	err = w.Sandbox.StartTask(sandboxTaskID)

	return sandboxTaskID, err
}

func (w *Worker) createTasksOnAllProjects(taskMessage *models.TaskMessageDTO) (err error) {
	projects, err := w.projectUsecase.ListByOrganizationID(w.ctx, taskMessage.OrganizationID)
	if err != nil {
		return err
	}
	var tasks []*models.TaskResponseDTO
	for _, project := range projects {
		matchTag := false
		for _, taskTag := range taskMessage.Tags {
			for _, projectTag := range project.Tags {
				if strings.EqualFold(taskTag, projectTag) {
					matchTag = true
					break
				}
			}
		}
		if !matchTag && len(taskMessage.Tags) > 0 {
			continue
		}
		task, err := w.taskUsecase.GetLastTemplate(w.ctx, project.ID)
		if err != nil {
			return err
		}
		if task != nil {
			tasks = append(tasks, task)
		}
	}

	for _, originalTask := range tasks {
		parameters := originalTask.Parameters
		for k, v := range taskMessage.Parameters {
			parameters[k] = v
		}

		taskID := uuid.Must(uuid.NewV4()).String()
		loc, _ := time.LoadLocation("Europe/Moscow")
		task := models.Task{
			TaskID:          taskID,
			OrganizationID:  originalTask.OrganizationID,
			ProjectID:       originalTask.ProjectID,
			Parameters:      parameters,
			StartTime:       time.Now().In(loc).Unix(),
			Status:          models.Created,
			Analysers:       taskMessage.Analysers,
			NonTemplateScan: false,
		}
		normalizedParameters, err := w.taskUsecase.NormalizeTaskParameters(parameters)
		if err != nil {
			simplelog.Info("failed to normalize task parameters", "err", err,
				"organizationID", originalTask.OrganizationID, "projectID", originalTask.ProjectID)
			continue
		}
		err = w.taskUsecase.Create(w.ctx, &task)
		if err != nil {
			simplelog.Info("failed to create task", "err", err,
				"organizationID", originalTask.OrganizationID, "projectID", originalTask.ProjectID)
			continue
		}

		msg, _ := json.Marshal(&models.TaskMessageDTO{
			OrganizationID: originalTask.OrganizationID,
			ProjectID:      originalTask.ProjectID,
			TaskID:         taskID,
			Parameters:     normalizedParameters,
			Analysers:      taskMessage.Analysers,
		})
		_, err = w.Queue.SendMessage(&queue.SendOptions{
			QueueURL: w.CFG.TasksQueueURL(),
			Msg:      string(msg),
		})
		if err != nil {
			simplelog.Error("failed to send message", "err", err)
		}
	}

	return nil
}

func (w *Worker) taskProcessor(wg *sync.WaitGroup, messages <-chan *sqs.Message) {
	wg.Add(1)
	defer wg.Done()

	for msg := range messages {
		var taskMessage models.TaskMessageDTO
		err := json.Unmarshal([]byte(*msg.Body), &taskMessage)
		if err != nil {
			simplelog.Error("failed to unmarshal task message", "err", err)
			err := w.Queue.DeleteMessage(&queue.DeleteOptions{
				QueueURL:      w.CFG.TasksQueueURL(),
				ReceiptHandle: msg.ReceiptHandle,
			})
			if err != nil {
				simplelog.Error("failed to delete message from queue", "err", err)
			}
			continue
		}

		simplelog.Info("processor", "task", taskMessage)

		if taskMessage.OnAllProjects {
			// if processing time will be greater than visibility timeout
			err := w.Queue.DeleteMessage(&queue.DeleteOptions{
				QueueURL:      w.CFG.TasksQueueURL(),
				ReceiptHandle: msg.ReceiptHandle,
			})
			if err != nil {
				simplelog.Error("failed to delete message from queue", "err", err)
			}

			err = w.createTasksOnAllProjects(&taskMessage)
			if err != nil {
				simplelog.Info("createTasksOnAllProjects error", "err", err)
			}
		} else {
			task := models.Task{TaskID: taskMessage.TaskID, Status: models.Pending}
			err := w.taskUsecase.Update(w.ctx, &task)
			if err != nil {
				simplelog.Info("failed to update task status", "err", err)
			}

			sandboxTaskID, err := w.startSandboxTask(&taskMessage)
			if err != nil {
				simplelog.Error("failed to start Sandbox task", "err", err)

				task := models.Task{TaskID: taskMessage.TaskID, SandboxTaskID: sandboxTaskID, Status: models.Failed}
				if err2 := w.taskUsecase.Update(w.ctx, &task); err2 != nil {
					simplelog.Error("failed to update task status", "err", err2)
				}

				switch err.(type) {
				case sandbox.TooManyRequests:
					time.Sleep(time.Duration(w.CFG.SandboxRetryTimeout) * time.Second)
					continue
				}
			} else {
				task := models.Task{TaskID: taskMessage.TaskID, SandboxTaskID: sandboxTaskID, Status: models.Running}
				if err := w.taskUsecase.Update(w.ctx, &task); err != nil {
					simplelog.Error("failed to update task status", "err", err)
				}
			}

			err = w.Queue.DeleteMessage(&queue.DeleteOptions{
				QueueURL:      w.CFG.TasksQueueURL(),
				ReceiptHandle: msg.ReceiptHandle,
			})
			if err != nil {
				simplelog.Error("failed to delete message from queue", "err", err)
			}
		}
	}
}
