package main

import (
	"context"
	"os"
	"os/signal"
	"syscall"
	"time"

	"a.yandex-team.ru/security/impulse/api/worker/internal/config"
	_infra "a.yandex-team.ru/security/impulse/api/worker/internal/infra"
	"a.yandex-team.ru/security/impulse/api/worker/internal/worker"
	"a.yandex-team.ru/security/libs/go/simplelog"
)

const (
	shutdownDeadline = time.Second * 300
)

func newConfig() *config.Config {
	var cfg *config.Config
	switch os.Getenv("IMPULSE_ENV") {
	case "dev":
		simplelog.SetLevel(simplelog.DebugLevel)
		cfg = &config.Config{
			Debug:                     true,
			DBConn:                    os.Getenv("DB_DSN"),
			DBPassword:                os.Getenv("DB_PASSWORD"),
			DBRetries:                 1,
			SqsEndpoint:               "https://sqs.yandex.net",
			SqsAccount:                "servicesecurity-dev",
			SqsRetries:                1,
			SqsOAuthToken:             os.Getenv("SQS_OAUTH_TOKEN"),
			S3Endpoint:                "https://s3.mdst.yandex.net",
			S3Region:                  "us-east-1",
			S3AccessKeyID:             "7RzzcRK5FBAHJhwMOwWZ",
			S3SecretAccessKey:         os.Getenv("S3_ACCESS_KEY"),
			Concurrency:               4,
			SandboxEndpoint:           "https://sandbox.yandex-team.ru",
			SandboxOAuthToken:         os.Getenv("SANDBOX_OAUTH_TOKEN"),
			SandboxRetryTimeout:       5,
			CodeQLBuildTaskInterval:   60 * 60 * 4,
			CodeQLBuildTaskBatchSize:  150,
			SendNotificationsInterval: 10000,
			SecNofityURL:              "http://secnotify",
			SecNotifyDstTvmID:         2017559,
			SecNofityToken:            os.Getenv("SECNOTIFY_TOKEN"),
			UseAuth:                   false,
		}
	case "test":
		simplelog.SetLevel(simplelog.DebugLevel)
		cfg = &config.Config{
			Debug:                     true,
			DBConn:                    "host=c-mdbfts3u6vt7aop9seqf.rw.db.yandex.net port=6432 dbname=impulse sslmode=verify-full user=impulse",
			DBPassword:                os.Getenv("DB_PASSWORD"),
			DBRetries:                 1,
			SqsEndpoint:               "https://sqs.yandex.net",
			SqsAccount:                "servicesecurity-test",
			SqsRetries:                5,
			SqsOAuthToken:             os.Getenv("SQS_OAUTH_TOKEN"),
			S3Endpoint:                "https://s3.mdst.yandex.net",
			S3Region:                  "us-east-1",
			S3AccessKeyID:             "7RzzcRK5FBAHJhwMOwWZ",
			S3SecretAccessKey:         os.Getenv("S3_ACCESS_KEY"),
			Concurrency:               4,
			SandboxEndpoint:           "https://sandbox.yandex-team.ru",
			SandboxOAuthToken:         os.Getenv("SANDBOX_OAUTH_TOKEN"),
			SandboxRetryTimeout:       5,
			CodeQLBuildTaskInterval:   60 * 60 * 4,
			CodeQLBuildTaskBatchSize:  150,
			SendNotificationsInterval: 600,
			SecNofityURL:              "http://secnotify.sec.yandex-team.ru",
			SecNotifyDstTvmID:         2017559,
			SecNofityToken:            os.Getenv("SECNOTIFY_TOKEN"),
			UseAuth:                   true,
		}
	case "prod":
		fallthrough
	default:
		cfg = &config.Config{
			Debug: false,
		}
	}

	return cfg
}

func main() {
	infra, err := _infra.New(newConfig())
	if err != nil {
		panic(err)
	}
	wrk := worker.New(infra)

	// Setup quit channel
	quit := make(chan os.Signal)

	// Start worker
	go func() {
		if err := wrk.Start(); err != nil {
			simplelog.Error("worker stopped with error", "err", err)
			// send signal manually, server failed to start
			signal.Stop(quit)
			quit <- syscall.SIGTERM
		}
	}()

	// Wait termination
	signal.Notify(quit, syscall.SIGINT, syscall.SIGTERM)
	<-quit

	ctx, cancel := context.WithTimeout(context.Background(), shutdownDeadline)
	defer cancel()

	if err := wrk.Shutdown(ctx); err != nil {
		simplelog.Error("failed to shutdown worker", "err", err)
	}
}
