package models

import (
	"database/sql/driver"
	"encoding/json"

	"github.com/jackc/pgtype"
	"golang.org/x/xerrors"
)

type (
	ProjectTags          []string
	NotificationSettings struct {
		Enabled   bool                   `json:"enabled,omitempty"`
		Filter    map[string]interface{} `json:"filter,omitempty"`
		Assignee  string                 `json:"assignee,omitempty"`
		Followers []string               `json:"followers,omitempty"`
	}

	Project struct {
		ID                   int                  `json:"id" db:"id"`
		Name                 string               `json:"name" db:"name"`
		Slug                 string               `json:"slug" db:"slug"`
		OrganizationID       int                  `json:"organization_id" db:"organization_id"`
		TrackerQueue         string               `json:"tracker_queue" db:"tracker_queue"`
		ABCServiceID         int                  `json:"abc_service_id" db:"abc_service_id"`
		Tags                 ProjectTags          `json:"tags" db:"tags"`
		NotificationSettings NotificationSettings `json:"notification_settings" db:"notification_settings"`
		BuildCodeQLIndex     bool                 `json:"build_codeql_index" db:"build_codeql_index"`
		Deleted              bool                 `db:"deleted"`
	}

	ProjectStatistics struct {
		TotalVulnerabilitiesCount               int `json:"total_vulnerabilities_count" db:"total_vulnerabilities_count"`
		TotalNotFalseVulnerabilitiesCount       int `json:"total_not_false_count" db:"total_not_false_count"`
		TotalNotReviewedVulnerabilitiesCount    int `json:"total_not_reviewed_count" db:"total_not_reviewed_count"`
		BlockerNotFalseVulnerabilitiesCount     int `json:"blocker_not_false_count" db:"blocker_not_false_count"`
		BlockerNotReviewedVulnerabilitiesCount  int `json:"blocker_not_reviewed_count" db:"blocker_not_reviewed_count"`
		CriticalNotFalseVulnerabilitiesCount    int `json:"critical_not_false_count" db:"critical_not_false_count"`
		CriticalNotReviewedVulnerabilitiesCount int `json:"critical_not_reviewed_count" db:"critical_not_reviewed_count"`
		MediumNotFalseVulnerabilitiesCount      int `json:"medium_not_false_count" db:"medium_not_false_count"`
		MediumNotReviewedVulnerabilitiesCount   int `json:"medium_not_reviewed_count" db:"medium_not_reviewed_count"`
		LowNotFalseVulnerabilitiesCount         int `json:"low_not_false_count" db:"low_not_false_count"`
		LowNotReviewedVulnerabilitiesCount      int `json:"low_not_reviewed_count" db:"low_not_reviewed_count"`
		InfoNotFalseVulnerabilitiesCount        int `json:"info_not_false_count" db:"info_not_false_count"`
		InfoNotReviewedVulnerabilitiesCount     int `json:"info_not_reviewed_count" db:"info_not_reviewed_count"`
	}

	ProjectInfo struct {
		Project
		Statistics ProjectStatistics `json:"statistics" db:"statistics"`
	}

	ProjectUpdateRequestDTO struct {
		Name                 string               `json:"name"`
		OrganizationID       int                  `json:"organization_id"`
		TrackerQueue         string               `json:"tracker_queue"`
		ABCServiceID         int                  `json:"abc_service_id"`
		Tags                 ProjectTags          `json:"tags"`
		NotificationSettings NotificationSettings `json:"notification_settings"`
		BuildCodeQLIndex     bool                 `json:"build_codeql_index"`
	}

	ProjectVulnerabilitiesDeduplicationResponseDTO struct {
		ID              int                                      `json:"id"`
		LastUpdateToken string                                   `json:"last_update_token"`
		Vulnerabilities []*VulnerabilityDeduplicationResponseDTO `json:"vulnerabilities"`
	}

	ProjectVulnerabilitiesDeduplicationRequestDTO struct {
		LastUpdateToken             string                                              `json:"last_update_token"`
		NewVulnerabilities          []*NewVulnerabilityDeduplicationRequestDTO          `json:"new_vulnerabilities"`
		DeduplicatedVulnerabilities []*DeduplicatedVulnerabilityDeduplicationRequestDTO `json:"deduplicated_vulnerabilities"`
		TaskID                      string                                              `json:"task_id"`
		RawReportURL                string                                              `json:"raw_report_url"`
		ReportURL                   string                                              `json:"report_url"`
		CommitHash                  string                                              `json:"commit_hash"`
		StartTime                   int64                                               `json:"start_time"`
		EndTime                     int64                                               `json:"end_time"`
	}
)

func (projectTags *ProjectTags) Scan(val interface{}) (err error) {
	switch v := val.(type) {
	case string:
		tags := pgtype.VarcharArray{}
		err = tags.Scan(v)
		if err != nil {
			return
		}
		err = tags.AssignTo(projectTags)
		return
	default:
		return nil
	}
}

func (projectTags ProjectTags) Value() (driver.Value, error) {
	tags := pgtype.VarcharArray{}
	_ = tags.Set(projectTags)
	return tags.Value()
}

func (filter NotificationSettings) Value() (driver.Value, error) {
	b, err := json.Marshal(filter)
	if err != nil {
		return "", err
	} else {
		return string(b), nil
	}
}

func (filter *NotificationSettings) Scan(val interface{}) (err error) {
	if val == nil {
		return
	}
	switch v := val.(type) {
	case []byte:
		err = json.Unmarshal(v, &filter)
		return
	default:
		return xerrors.Errorf("Unsupported type: %T", v)
	}
}
