package models

import (
	"database/sql/driver"
	"encoding/json"

	"golang.org/x/xerrors"
)

type ScanTypeName string

const (
	BANDIT      ScanTypeName = "bandit_scan"
	BANDIT2     ScanTypeName = "bandit2_scan"
	BANDIT3     ScanTypeName = "bandit3_scan"
	ESLINT      ScanTypeName = "eslint_scan"
	FINDSECBUGS ScanTypeName = "findsecbugs_scan"
	YADI        ScanTypeName = "yadi_scan"
	RIPS        ScanTypeName = "rips_scan"
	GOSEC       ScanTypeName = "gosec_scan"
	YODAX       ScanTypeName = "yodax_scan"
	CODEQL      ScanTypeName = "codeql_scan"
	SEMGREP     ScanTypeName = "semgrep_scan"
	BLACKDUCK   ScanTypeName = "blackduck_scan"
)

type (
	ScanType struct {
		ID       int          `json:"id" db:"id"`
		TypeName ScanTypeName `json:"type_name" db:"type_name"`
		Title    string       `json:"title" db:"title"`
	}

	ScanParameterProperties map[string]string
	ScanParameter           struct {
		ScanTypeID           int                     `json:"scan_type_id" db:"scan_type_id"`
		Name                 string                  `json:"name" db:"name"`
		Label                string                  `json:"label" db:"label"`
		Component            string                  `json:"component" db:"component"`
		Properties           ScanParameterProperties `json:"properties" db:"properties"`
		NonTemplateParameter bool                    `json:"non_template_parameter" db:"non_template_parameter"`
	}

	ScanTypeResponseDTO struct {
		ScanType
		Parameters []*ScanParameter `json:"parameters"`
	}
)

func (t ScanParameterProperties) Value() (driver.Value, error) {
	b, err := json.Marshal(t)
	if err != nil {
		return "", err
	} else {
		return string(b), nil
	}
}

func (t *ScanParameterProperties) Scan(value interface{}) error {
	switch v := value.(type) {
	case []byte:
		return json.Unmarshal(v, &t)
	case nil:
		return nil
	default:
		return xerrors.Errorf("unsupported type: %T", v)
	}
}
