package models

import (
	"database/sql/driver"
	"encoding/json"

	"golang.org/x/xerrors"
)

type TaskStatus int

const (
	Created TaskStatus = iota
	Pending
	Running
	Failed
	Finished
	FinishedCallbackFailed
)

type (
	TaskParameters map[string]interface{}
	TaskAnalysers  []string

	// swagger:model
	TaskStatusDTO struct {
		// Required: true
		Status TaskStatus `json:"status" form:"status" query:"status"`
	}

	// swagger:model
	TaskRequestDTO struct {
		// Required: true
		OrganizationID int `json:"organization_id" form:"organization_id" query:"organization_id"`
		// Required: true
		ProjectID       int            `json:"project_id" form:"project_id" query:"project_id"`
		Parameters      TaskParameters `json:"parameters" form:"parameters" query:"parameters"`
		Analysers       TaskAnalysers  `json:"analysers" form:"analysers" query:"analysers"`
		CallbackURL     string         `json:"callback_url" form:"callback_url" query:"callback_url"`
		NonTemplateScan bool           `json:"non_template_scan" form:"non_template_scan" query:"non_template_scan"`
	}

	TaskOrganizationRequestDTO struct {
		Parameters TaskParameters `json:"parameters" form:"parameters" query:"parameters"`
		Analysers  TaskAnalysers  `json:"analysers" form:"analysers" query:"analysers"`
		Tags       ProjectTags    `json:"tags" form:"tags" query:"tags"`
	}

	TaskRequestFromSandboxDTO struct {
		OrganizationID  int            `json:"organization_id" form:"organization_id" query:"organization_id"`
		ProjectID       int            `json:"project_id" form:"project_id" query:"project_id"`
		Parameters      TaskParameters `json:"parameters" form:"parameters" query:"parameters"`
		Analysers       TaskAnalysers  `json:"analysers" form:"analysers" query:"analysers"`
		SandboxTaskID   int            `json:"sandbox_task_id" form:"sandbox_task_id" query:"sandbox_task_id"`
		NonTemplateScan bool           `json:"non_template_scan" form:"non_template_scan" query:"non_template_scan"`
	}

	TaskMessageDTO struct {
		OrganizationID int            `json:"organization_id"`
		ProjectID      int            `json:"project_id,omitempty"`
		TaskID         string         `json:"task_id,omitempty"`
		Parameters     TaskParameters `json:"parameters"`
		Analysers      TaskAnalysers  `json:"analysers"`
		OnAllProjects  bool           `json:"on_all_projects"`
		Tags           ProjectTags    `json:"tags"`
	}

	Task struct {
		TaskID             string         `json:"task_id" db:"task_id"`
		OrganizationID     int            `json:"organization_id" db:"organization_id"`
		ProjectID          int            `json:"project_id" db:"project_id"`
		WorkflowID         string         `json:"workflow_id,omitempty" db:"workflow_id"`
		WorkflowInstanceID string         `json:"workflow_instance_id,omitempty" db:"workflow_instance_id"`
		Parameters         TaskParameters `json:"parameters" db:"parameters"`
		Analysers          TaskAnalysers  `json:"analysers" db:"analysers"`
		SandboxTaskID      int            `json:"sandbox_task_id" db:"sandbox_task_id"`
		StartTime          int64          `json:"start_time" db:"start_time"`
		EndTime            int64          `json:"end_time,omitempty" db:"end_time"`
		Status             TaskStatus     `json:"status" db:"status"`
		CronID             int            `json:"cron_id,omitempty" db:"cron_id"`
		CallbackURL        string         `json:"callback_url,omitempty" db:"callback_url"`
		NonTemplateScan    bool           `json:"non_template_scan,omitempty" db:"non_template_scan"`
	}

	TaskInfo struct {
		Task                   Task                   `json:"task"`
		ScanInstanceStatistics ScanInstanceStatistics `json:"statistics"`
	}

	TaskResponseDTO struct {
		Task
		Workflow    string `json:"workflow" db:"workflow"`
		ProjectName string `json:"project_name,omitempty" db:"project_name"`
		Error       string `json:"error,omitempty"`
	}

	TasksListResponseDTO struct {
		Total int                `json:"total"`
		Tasks []*TaskResponseDTO `json:"tasks"`
	}

	TaskAnalysersRequestDTO struct {
		Analysers TaskAnalysers `json:"analysers"`
	}
)

func (t TaskAnalysers) Value() (driver.Value, error) {
	b, err := json.Marshal(t)
	if err != nil {
		return "", err
	} else {
		return string(b), nil
	}
}

func (t *TaskAnalysers) Scan(value interface{}) error {
	switch v := value.(type) {
	case []byte:
		return json.Unmarshal(v, &t)
	case nil:
		return nil
	default:
		return xerrors.Errorf("unsupported type: %T", v)
	}
}

func (t TaskParameters) Value() (driver.Value, error) {
	b, err := json.Marshal(t)
	if err != nil {
		return "", err
	} else {
		return string(b), nil
	}
}

func (t *TaskParameters) Scan(value interface{}) error {
	switch v := value.(type) {
	case []byte:
		return json.Unmarshal(v, &t)
	case nil:
		return nil
	default:
		return xerrors.Errorf("unsupported type: %T", v)
	}
}
