package models

import (
	"database/sql/driver"
	"encoding/json"
	"time"

	"golang.org/x/xerrors"
)

// Vulnerability Severity 'blocker', 'critical', 'medium', 'low', 'info'
type SeverityType string

const (
	Info     SeverityType = "info"
	Low      SeverityType = "low"
	Medium   SeverityType = "medium"
	Critical SeverityType = "critical"
	Blocker  SeverityType = "blocker"
)

type ConfidenceType string

const (
	ConfidenceHigh         ConfidenceType = "high"
	ConfidenceMedium       ConfidenceType = "medium"
	ConfidenceLow          ConfidenceType = "low"
	ConfidenceExperimental ConfidenceType = "experimental"
	ConfidenceIgnore       ConfidenceType = "ignore"
	ConfidenceUnknown      ConfidenceType = "unknown"
)

func (s *SeverityType) Scan(value interface{}) error {
	asString, ok := value.(string)
	if !ok {
		return xerrors.Errorf("Scan source is not string, but %T", value)
	}
	*s = SeverityType(asString)
	return nil
}

func (s SeverityType) Value() (driver.Value, error) {
	return string(s), nil
}

func (s SeverityType) MarshalJSON() ([]byte, error) {
	return json.Marshal(string(s))
}

// Vulnerability Status Type
type StatusType string

const (
	NotReviewed    StatusType = "not_reviewed"
	ToVerify       StatusType = "to_verify"
	Confirmed      StatusType = "confirmed"
	NotExploitable StatusType = "not_exploitable"
	NotAnIssue     StatusType = "not_an_issue"
)

func (s *StatusType) Scan(value interface{}) error {
	asString, ok := value.(string)
	if !ok {
		return xerrors.Errorf("Scan source is not string, but %T", value)
	}
	*s = StatusType(asString)
	return nil
}

func (s StatusType) Value() (driver.Value, error) {
	return string(s), nil
}

func (s StatusType) MarshalJSON() ([]byte, error) {
	return json.Marshal(string(s))
}

type (
	VulnerabilityProperties map[string]interface{}

	Vulnerability struct {
		ID                int                     `json:"id" form:"id" query:"id" db:"id"`
		ProjectID         int                     `json:"project_id" db:"project_id"`
		ProjectName       string                  `json:"project_name" db:"project_name"`
		ScanID            int                     `json:"scan_id" form:"scan_id" query:"scan_id" db:"scan_id"`
		ScanInstanceID    int                     `json:"scan_instance_id" db:"scan_instance_id"`
		ScanTypeName      string                  `json:"scan_type_name" db:"scan_type_name"`
		ScanTypeTitle     string                  `json:"scan_type_title" db:"scan_type_title"`
		Severity          *SeverityType           `json:"severity,omitempty" db:"severity" sql:"type:severity_type"`
		Status            *StatusType             `json:"status,omitempty" db:"status" sql:"type:status_type"`
		CategoryID        int                     `json:"category_id" form:"category_id" query:"category_id" db:"category_id"`
		CategoryName      string                  `json:"category_name" db:"category_name"`
		FirstFoundAt      time.Time               `json:"first_found_at" db:"first_found_at"`
		KeyProperties     VulnerabilityProperties `json:"key_properties" form:"key_properties" query:"key_properties" db:"key_properties"`
		DisplayProperties VulnerabilityProperties `json:"display_properties" form:"display_properties" query:"display_properties" db:"display_properties"`
		TrackerTicket     string                  `json:"tracker_ticket" db:"tracker_ticket"`
	}

	VulnerabilityDeduplicationResponseDTO struct {
		ID            int                     `json:"id" form:"id" query:"id" db:"id"`
		KeyProperties VulnerabilityProperties `json:"key_properties" form:"key_properties" query:"key_properties" db:"key_properties"`
	}

	NewVulnerabilityDeduplicationRequestDTO struct {
		Severity          SeverityType            `json:"severity,omitempty" db:"severity" sql:"type:severity_type"`
		Category          string                  `json:"category" db:"category"`
		KeyProperties     VulnerabilityProperties `json:"key_properties" form:"key_properties" query:"key_properties" db:"key_properties"`
		DisplayProperties VulnerabilityProperties `json:"display_properties" form:"display_properties" query:"display_properties" db:"display_properties"`
	}

	DeduplicatedVulnerabilityDeduplicationRequestDTO struct {
		ID                int                     `json:"id" form:"id" query:"id" db:"id"`
		Severity          SeverityType            `json:"severity,omitempty" db:"severity" sql:"type:severity_type"`
		Category          string                  `json:"category" db:"category"`
		KeyProperties     VulnerabilityProperties `json:"key_properties" form:"key_properties" query:"key_properties" db:"key_properties"`
		DisplayProperties VulnerabilityProperties `json:"display_properties" form:"display_properties" query:"display_properties" db:"display_properties"`
	}

	VulnerabilityUpdateStatusRequestDTO struct {
		Status StatusType `json:"status"`
	}
	BatchUpdateVulnerabilityStatusRequestDTO struct {
		Status          StatusType `json:"status"`
		Vulnerabilities []int      `json:"vulnerabilities"`
	}

	VulnerabilityUpdateTrackerTicketRequestDTO struct {
		TrackerTicket string `json:"tracker_ticket"`
	}

	VulnerabilityCreateTrackerTicketRequestDTO struct {
		TrackerQueue string   `json:"tracker_queue"`
		Title        string   `json:"title"`
		Description  string   `json:"description"`
		Followers    []string `json:"followers"`
	}

	VulnerabilityCreateTrackerTicketResponseDTO struct {
		TrackerTicket string `json:"tracker_ticket"`
	}
)

func (vulnProperties *VulnerabilityProperties) Scan(val interface{}) (err error) {
	switch v := val.(type) {
	case []byte:
		err = json.Unmarshal(v, &vulnProperties)
		return
	default:
		return xerrors.Errorf("Unsupported type: %T", v)
	}
}
