package queue

import (
	"time"

	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/aws/credentials"
	"github.com/aws/aws-sdk-go/aws/session"
	"github.com/aws/aws-sdk-go/service/sqs"

	"a.yandex-team.ru/library/go/core/xerrors"
	"a.yandex-team.ru/library/go/yandex/awstvm"
	"a.yandex-team.ru/security/libs/go/yahttp"
)

type queueImpl struct {
	sqs *sqs.SQS
}

func New(opts *Options) (*queueImpl, error) {
	if opts.AuthType == 0 || opts.AuthType == AuthAuto {
		switch {
		case opts.OAuthToken != "":
			opts.AuthType = AuthOAuth
		case opts.TVMClent != nil:
			opts.AuthType = AuthTvm
		default:
			return nil, xerrors.Errorf("failed to determine auth type")
		}
	}

	var sqsCred *credentials.Credentials
	switch opts.AuthType {
	case AuthOAuth:
		sqsCred = credentials.NewStaticCredentials(
			opts.Account,
			"unused",
			opts.OAuthToken,
		)
	case AuthTvm:
		cred, err := awstvm.NewSqsCredentials(opts.TVMClent, opts.Account)
		if err != nil {
			return nil, xerrors.Errorf("failed to create SQS TVM credentials: %w", err)
		}
		sqsCred = cred
	}

	maxRetries := 3
	if opts.MaxRetries != 0 {
		maxRetries = opts.MaxRetries
	}

	sess, err := session.NewSession(&aws.Config{
		Region:      aws.String(SqsRegion),
		Endpoint:    &opts.Endpoint,
		Credentials: sqsCred,
		MaxRetries:  &maxRetries,
		HTTPClient: yahttp.NewClient(yahttp.Config{
			RedirectPolicy: yahttp.RedirectNoFollow,
			DialTimeout:    time.Second,
			Timeout:        time.Minute * 2,
		}),
	})

	if err != nil {
		return nil, xerrors.Errorf("failed to create SQS session: %w", err)
	}

	return &queueImpl{
		sqs: sqs.New(sess),
	}, nil
}

func (q *queueImpl) SendMessage(opts *SendOptions) (msgID string, resultErr error) {
	if len(opts.Msg) > MaxMessageSize {
		resultErr = xerrors.Errorf("maximum message size exceed: %d > %d", len(opts.Msg), MaxMessageSize)
		return
	}

	result, err := q.sqs.SendMessage(&sqs.SendMessageInput{
		MessageBody:    &opts.Msg,
		QueueUrl:       &opts.QueueURL,
		MessageGroupId: &opts.MessageGroupID,
	})

	if err != nil {
		resultErr = xerrors.Errorf("failed to send message: %w", err)
		return
	}

	msgID = *result.MessageId
	return
}

func (q *queueImpl) ReceiveMessage(opts *ReceiveOptions) (msgs []*sqs.Message, resultErr error) {
	maxMsg := DefaultMaxMsg
	if opts.MaxNumberOfMessages > 0 {
		maxMsg = opts.MaxNumberOfMessages
	}

	pollWaitSeconds := DefaultPollSeconds
	if opts.WaitTimeSeconds > 0 {
		pollWaitSeconds = opts.WaitTimeSeconds
	}

	result, err := q.sqs.ReceiveMessage(&sqs.ReceiveMessageInput{
		QueueUrl:            &opts.QueueURL,
		MaxNumberOfMessages: &maxMsg,
		WaitTimeSeconds:     &pollWaitSeconds,
	})

	if err != nil {
		resultErr = xerrors.Errorf("failed to fetch sqs message: %w", err)
		return
	}

	msgs = result.Messages
	return
}

func (q *queueImpl) DeleteMessage(opts *DeleteOptions) error {
	_, err := q.sqs.DeleteMessage(&sqs.DeleteMessageInput{
		QueueUrl:      &opts.QueueURL,
		ReceiptHandle: opts.ReceiptHandle,
	})

	if err != nil {
		return xerrors.Errorf("failed to delete sqs message: %w", err)
	}
	return nil
}

func (q *queueImpl) ChangeMessageVisibility(opts *ChangeMessageVisibilityOptions) error {
	_, err := q.sqs.ChangeMessageVisibility(&sqs.ChangeMessageVisibilityInput{
		ReceiptHandle:     opts.ReceiptHandle,
		QueueUrl:          &opts.QueueURL,
		VisibilityTimeout: &opts.VisibilityTimeout,
	})

	if err != nil {
		return xerrors.Errorf("failed to delete sqs message: %w", err)
	}
	return nil
}
