package queue

import (
	"github.com/aws/aws-sdk-go/service/sqs"

	"a.yandex-team.ru/library/go/yandex/tvm"
)

const (
	SqsRegion                = "yandex"
	MaxMessageSize           = 250 * 1000
	DefaultMaxMsg      int64 = 1
	DefaultPollSeconds int64 = 20
)

type AuthType int

const (
	AuthAuto AuthType = iota
	AuthOAuth
	AuthTvm
)

type (
	Queue interface {
		SendMessage(opts *SendOptions) (msgID string, resultErr error)
		ReceiveMessage(opts *ReceiveOptions) (msgs []*sqs.Message, resultErr error)
		DeleteMessage(opts *DeleteOptions) error
		ChangeMessageVisibility(opts *ChangeMessageVisibilityOptions) error
	}

	Options struct {
		// SQS upstream url
		Endpoint string

		// Auth type
		AuthType AuthType

		// SQS Account
		Account string

		// OAuth token, used by auth type AuthOAuth
		OAuthToken string

		// TVM Client, used by auth type AuthTVM
		TVMClent tvm.Client

		// The maximum number of times that a request will be retried for failures.
		// Default: 3
		MaxRetries int
	}

	SendOptions struct {
		// SQS queue
		QueueURL string

		// Message to send
		Msg string

		// This parameter applies only to FIFO (first-in-first-out) queues
		MessageGroupID string
	}

	ReceiveOptions struct {
		// SQS queue
		QueueURL string

		// The maximum number of messages to return. SQS never returns more messages
		// than this value (however, fewer messages might be returned).
		// Valid values: 1 to 10
		// Default: 1
		MaxNumberOfMessages int64

		// The duration (in seconds) for which the call waits for a message to arrive
		// in the queue before returning. If a message is available, the call returns
		// sooner than WaitTimeSeconds. If no messages are available and the wait time
		// expires, the call returns successfully with an empty list of messages.
		WaitTimeSeconds int64
	}

	DeleteOptions struct {
		// SQS queue
		QueueURL string

		// The receipt handle associated with the message to delete.
		ReceiptHandle *string
	}

	ChangeMessageVisibilityOptions struct {
		// SQS queue
		QueueURL string

		// The receipt handle associated with the message whose visibility timeout is
		// changed. This parameter is returned by the ReceiveMessage action.
		ReceiptHandle *string

		// The new value for the message's visibility timeout (in seconds).
		// Values values: 0 to 43200. Maximum: 12 hours.
		VisibilityTimeout int64
	}
)
