package s3

import (
	"os"
	"strings"

	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/aws/credentials"
	"github.com/aws/aws-sdk-go/aws/session"
	"github.com/aws/aws-sdk-go/service/s3"
	"github.com/aws/aws-sdk-go/service/s3/s3manager"
)

type (
	S3Uploader interface {
		S3Upload(bucket, key, path string) error
		S3UploadString(bucket, key, data string) (string, error)
	}
	S3Downloader interface {
		S3Download(bucket, key string) ([]byte, error)
		FileSize(bucket, key string) (int64, error)
	}

	S3UploaderClient struct {
		uploader *s3manager.Uploader
	}

	S3DownloaderClient struct {
		s3         *s3.S3
		downloader *s3manager.Downloader
	}

	Options struct {
		Endpoint        string
		Region          string
		AccessKeyID     string
		SecretAccessKey string
	}
)

func newSession(opts *Options) (*session.Session, error) {
	accessKeyID := opts.AccessKeyID
	secretAccessKey := opts.SecretAccessKey

	var config aws.Config
	if accessKeyID != "" && secretAccessKey != "" {
		config = aws.Config{
			Endpoint:    aws.String(opts.Endpoint),
			Region:      aws.String(opts.Region),
			Credentials: credentials.NewStaticCredentials(accessKeyID, secretAccessKey, ""),
		}
	} else {
		config = aws.Config{
			Endpoint:    aws.String(opts.Endpoint),
			Region:      aws.String(opts.Region),
			Credentials: credentials.AnonymousCredentials,
		}
	}

	return session.NewSession(&config)
}

func NewUploader(opts *Options) (S3Uploader, error) {
	sess, err := newSession(opts)
	if err != nil {
		return nil, err
	}
	uploader := s3manager.NewUploader(sess)
	return &S3UploaderClient{uploader: uploader}, nil
}

func (s3u *S3UploaderClient) S3Upload(bucket, key, path string) error {
	fobj, err := os.Open(path)
	if err != nil {
		return err
	}
	upParams := &s3manager.UploadInput{
		Bucket: &bucket,
		Key:    &key,
		Body:   fobj,
	}
	_, err = s3u.uploader.Upload(upParams, func(u *s3manager.Uploader) {
		u.PartSize = 10 * 1024 * 1024
		u.LeavePartsOnError = true
	})

	return err
}

func (s3u *S3UploaderClient) S3UploadString(bucket, key, data string) (string, error) {
	reader := strings.NewReader(data)

	output, err := s3u.uploader.Upload(&s3manager.UploadInput{
		Bucket: aws.String(bucket),
		Key:    aws.String(key),
		Body:   reader,
	})
	if err != nil {
		return "", err
	}

	return output.Location, nil
}

func NewDownloader(opts *Options) (S3Downloader, error) {
	sess, err := newSession(opts)
	if err != nil {
		return nil, err
	}
	s3 := s3.New(sess)
	downloader := s3manager.NewDownloader(sess)
	return &S3DownloaderClient{s3: s3, downloader: downloader}, nil
}

func (s3d *S3DownloaderClient) S3Download(bucket, key string) ([]byte, error) {
	content := aws.NewWriteAtBuffer([]byte{})
	_, err := s3d.downloader.Download(content, &s3.GetObjectInput{
		Bucket: aws.String(bucket),
		Key:    aws.String(key),
	})
	if err != nil {
		return nil, err
	}
	return content.Bytes(), nil
}

func (s3d *S3DownloaderClient) FileSize(bucket, key string) (int64, error) {
	output, err := s3d.s3.HeadObject(&s3.HeadObjectInput{
		Bucket: aws.String(bucket),
		Key:    aws.String(key),
	})
	if err != nil {
		return 0, err
	}
	return aws.Int64Value(output.ContentLength), nil

}
