package api

import (
	"context"
	"errors"
	"fmt"
	"strconv"
	"strings"

	"github.com/go-resty/resty/v2"

	"a.yandex-team.ru/library/go/core/xerrors"
	"a.yandex-team.ru/library/go/yandex/tvm"
	tvm_api "a.yandex-team.ru/library/go/yandex/tvm/tvmauth"
	"a.yandex-team.ru/security/impulse/models"
)

type (
	ImpulseAPI struct {
		Client     *resty.Client
		TvmClient  *tvm_api.Client
		TvmID      tvm.ClientID
		OAuthToken string
	}

	Options struct {
		Endpoint   string
		TvmID      tvm.ClientID
		OAuthToken string
	}

	apiResponse struct {
		Ok     bool        `json:"ok"`
		Result interface{} `json:"result,omitempty"`
		Error  string      `json:"error,omitempty"`
	}

	apiGetVulnerabilitiesResponse struct {
		Ok     bool                                                  `json:"ok"`
		Result models.ProjectVulnerabilitiesDeduplicationResponseDTO `json:"result,omitempty"`
		Error  string                                                `json:"error,omitempty"`
	}

	apiGetCodeqlIndex struct {
		Ok     bool                  `json:"ok"`
		Result models.CodeQLDatabase `json:"result,omitempty"`
		Error  string                `json:"error,omitempty"`
	}
)

func New(tvmClient *tvm_api.Client, opts *Options) *ImpulseAPI {
	httpc := resty.New().
		SetHeader("Content-Type", "application/json; charset=utf-8").
		SetBaseURL(opts.Endpoint)

	return &ImpulseAPI{
		Client:     httpc,
		TvmClient:  tvmClient,
		TvmID:      opts.TvmID,
		OAuthToken: opts.OAuthToken,
	}
}

func (api *ImpulseAPI) Close() error {
	return nil
}

func (api *ImpulseAPI) GetVulnerabilities(organizationID, projectID int, scanType models.ScanTypeName) (*models.ProjectVulnerabilitiesDeduplicationResponseDTO, error) {
	var authHeader, credential string
	if api.OAuthToken != "" {
		authHeader = "Authorization"
		credential = fmt.Sprintf("OAuth %s", api.OAuthToken)
	} else if api.TvmClient != nil {
		authHeader = "X-Ya-Service-Ticket"
		var err error
		credential, err = api.TvmClient.GetServiceTicketForID(context.Background(), api.TvmID)
		if err != nil {
			return nil, xerrors.Errorf("failed to receive TVM ticket for %d: %w", api.TvmID, err)
		}
	} else {
		return nil, xerrors.Errorf("use TVM or OAuth for request authorization")
	}

	var resp apiGetVulnerabilitiesResponse
	httpResp, err := api.Client.R().
		SetHeader(authHeader, credential).
		SetResult(&resp).
		SetPathParams(map[string]string{
			"orgID":    strconv.Itoa(organizationID),
			"projID":   strconv.Itoa(projectID),
			"scanType": fmt.Sprint(scanType),
		}).
		Get("/api/v1/storage/organizations/{orgID}/projects/{projID}/vulnerabilities/{scanType}")

	if err != nil {
		return nil, err
	}

	if !httpResp.IsSuccess() {
		return nil, WrongStatusCode{Status: httpResp.Status()}
	}

	if !resp.Ok {
		return nil, errors.New(resp.Error)
	}

	return &resp.Result, nil
}

func (api *ImpulseAPI) StoreVulnerabilities(organizationID, projectID int, scanType models.ScanTypeName, request *models.ProjectVulnerabilitiesDeduplicationRequestDTO) error {
	var authHeader, credential string
	if api.OAuthToken != "" {
		authHeader = "Authorization"
		credential = fmt.Sprintf("OAuth %s", api.OAuthToken)
	} else if api.TvmClient != nil {
		authHeader = "X-Ya-Service-Ticket"
		var err error
		credential, err = api.TvmClient.GetServiceTicketForID(context.Background(), api.TvmID)
		if err != nil {
			return err
		}
	} else {
		return xerrors.Errorf("use TVM or OAuth for request authorization")
	}

	var resp apiResponse
	httpResp, err := api.Client.R().
		SetHeader(authHeader, credential).
		SetResult(&resp).
		SetBody(request).
		SetPathParams(map[string]string{
			"orgID":    strconv.Itoa(organizationID),
			"projID":   strconv.Itoa(projectID),
			"scanType": fmt.Sprint(scanType),
		}).
		Post("/api/v1/storage/organizations/{orgID}/projects/{projID}/vulnerabilities/{scanType}")

	if err != nil {
		return err
	}

	if !httpResp.IsSuccess() {
		return WrongStatusCode{Status: httpResp.Status()}
	}

	if !resp.Ok {
		switch {
		case strings.HasPrefix(resp.Error, "lastUpdateToken Expired"):
			return LastUpdateTokenExpired{request.LastUpdateToken}
		default:
			return errors.New(resp.Error)
		}
	}

	return nil
}

func (api *ImpulseAPI) UpdateStatus(taskID string, status models.TaskStatus) error {
	var authHeader, credential string
	if api.OAuthToken != "" {
		authHeader = "Authorization"
		credential = fmt.Sprintf("OAuth %s", api.OAuthToken)
	} else if api.TvmClient != nil {
		authHeader = "X-Ya-Service-Ticket"
		var err error
		credential, err = api.TvmClient.GetServiceTicketForID(context.Background(), api.TvmID)
		if err != nil {
			return xerrors.Errorf("failed to receive TVM ticket for %d: %w", api.TvmID, err)
		}
	} else {
		return xerrors.Errorf("use TVM or OAuth for request authorization")
	}

	var resp apiResponse
	httpResp, err := api.Client.R().
		SetHeader(authHeader, credential).
		SetResult(&resp).
		SetBody(models.TaskStatusDTO{Status: status}).
		SetPathParams(map[string]string{
			"taskID": taskID,
		}).
		Post("/api/v1/control/tasks/{taskID}/status")

	if err != nil {
		return err
	}

	if !httpResp.IsSuccess() {
		return WrongStatusCode{Status: httpResp.Status()}
	}

	if !resp.Ok {
		return errors.New(resp.Error)
	}
	return nil
}

func (api *ImpulseAPI) GetCodeqlIndex(organizationID, projectID int, language, tag string) (*models.CodeQLDatabase, error) {
	var authHeader, credential string
	if api.OAuthToken != "" {
		authHeader = "Authorization"
		credential = fmt.Sprintf("OAuth %s", api.OAuthToken)
	} else if api.TvmClient != nil {
		authHeader = "X-Ya-Service-Ticket"
		var err error
		credential, err = api.TvmClient.GetServiceTicketForID(context.Background(), api.TvmID)
		if err != nil {
			return nil, xerrors.Errorf("failed to receive TVM ticket for %d: %w", api.TvmID, err)
		}
	} else {
		return nil, xerrors.Errorf("use TVM or OAuth for request authorization")
	}

	var resp apiGetCodeqlIndex
	httpResp, err := api.Client.R().
		SetHeader(authHeader, credential).
		SetResult(&resp).
		SetPathParams(map[string]string{
			"orgID":  strconv.Itoa(organizationID),
			"projID": strconv.Itoa(projectID),
			"lang":   language,
			"tag":    tag,
		}).
		Get("api/v1/codeql/organizations/{orgID}/projects/{projID}/languages/{lang}/tags/{tag}")

	if err != nil {
		return nil, err
	}

	if !httpResp.IsSuccess() {
		return nil, WrongStatusCode{Status: httpResp.Status()}
	}

	if !resp.Ok {
		return nil, errors.New(resp.Error)
	}

	return &resp.Result, nil
}
