package checkout

import (
	"fmt"
	"net/url"
	"path"
	"strings"

	"golang.org/x/xerrors"
)

type (
	Folder struct {
		Repository string `json:"repository"`
		Branch     string `json:"branch,omitempty"`
		Revision   string `json:"revision,omitempty"`
		Path       string `json:"path,omitempty"`
		IsArcadia  bool   `json:"is_arcadia"`
	}

	Checkout struct {
		Path    string            `json:"path"`
		Folders map[string]Folder `json:"folders"`
	}
)

func GenerateFileURL(info Folder, filePath string) (fileURL string, err error) {
	pathParts := strings.SplitN(filePath, "/", 2)
	if len(pathParts) == 2 {
		filePath = pathParts[1]
	}
	var u *url.URL
	u, err = url.Parse(info.Repository)
	if err != nil {
		return
	}
	host := u.Host

	switch host {
	case "github.yandex-team.ru", "git.adfox.ru", "github.com":
		u.Scheme = "https"
		u.User = nil
		u.Path = strings.TrimSuffix(u.Path, ".git")
		u.Path = path.Join(u.Path, "blob", info.Revision, info.Path, filePath)
		fileURL = u.String()
	case "bb.yandex-team.ru", "bitbucket.sdc.yandex-team.ru", "bitbucket.browser.yandex-team.ru":
		u.Path = strings.TrimSuffix(u.Path, ".git")
		parts := strings.Split(strings.TrimLeft(u.Path, "/"), "/")
		project := parts[0]
		repo := parts[1]
		u.Scheme = "https"
		u.User = nil
		u.Path = path.Join("projects", project, "repos", repo, "browse", info.Path, filePath)
		u.RawQuery = fmt.Sprintf("at=%s", info.Revision)
		fileURL = u.String()
	case "arcadia.yandex.ru":
		repoPath := strings.Replace(u.Path, "arc/trunk/arcadia", "arc_vcs", 1)
		fileURL = fmt.Sprintf("https://a.yandex-team.ru%s/%s?rev=%s", repoPath, filePath, info.Revision)
	case "git.edadeal.yandex-team.ru":
		u.Scheme = "https"
		u.User = nil
		u.Host = "gitlab.edadeal.yandex-team.ru"
		u.Path = strings.TrimSuffix(u.Path, ".git")
		u.Path = path.Join(u.Path, "blob", info.Revision, info.Path, filePath)
		fileURL = u.String()
	default:
		err = xerrors.New("repository not supported")
	}
	return
}

func GenerateFileURLWithLineNumber(info Folder, filePath string, lineNumber int) (fileURL string, err error) {
	pathParts := strings.SplitN(filePath, "/", 2)
	if len(pathParts) == 2 {
		filePath = pathParts[1]
	}
	var u *url.URL
	u, err = url.Parse(info.Repository)
	if err != nil {
		return
	}
	host := u.Host

	switch host {
	case "github.yandex-team.ru", "git.adfox.ru", "github.com", "gitlab.edadeal.yandex-team.ru":
		u.Scheme = "https"
		u.User = nil
		u.Path = strings.TrimSuffix(u.Path, ".git")
		u.Path = path.Join(u.Path, "blob", info.Revision, info.Path, filePath)
		u.Fragment = fmt.Sprintf("L%d", lineNumber)

		fileURL = u.String()
	case "bb.yandex-team.ru", "bitbucket.sdc.yandex-team.ru":
		u.Path = strings.TrimSuffix(u.Path, ".git")
		parts := strings.Split(strings.TrimLeft(u.Path, "/"), "/")
		project := parts[0]
		repo := parts[1]
		bbURL := url.URL{}
		bbURL.Scheme = "https"
		bbURL.Host = host
		bbURL.Path = path.Join("projects", project, "repos", repo, "browse", info.Path, filePath)
		bbURL.RawQuery = fmt.Sprintf("at=%s", info.Revision)
		bbURL.Fragment = fmt.Sprintf("%d", lineNumber)

		fileURL = bbURL.String()
	case "arcadia.yandex.ru":
		repoPath := strings.Replace(u.Path, "arc/trunk/arcadia", "arc_vcs", 1)
		fileURL = fmt.Sprintf("https://a.yandex-team.ru%s/%s?rev=%s#L%d", repoPath, filePath, info.Revision, lineNumber)
	case "git.edadeal.yandex-team.ru":
		u.Scheme = "https"
		u.User = nil
		u.Host = "gitlab.edadeal.yandex-team.ru"
		u.Path = strings.TrimSuffix(u.Path, ".git")
		u.Path = path.Join(u.Path, "blob", info.Revision, info.Path, filePath)
		u.Fragment = fmt.Sprintf("L%d", lineNumber)

		fileURL = u.String()
	default:
		err = xerrors.New("repository not supported")
	}
	return
}

func GenerateFileURLWithLineNumberByFolders(folders map[string]Folder, filePath string, lineNumber int) (fileURL string, err error) {
	if filePath == "" {
		err = xerrors.New("empty file path")
		return
	}
	fullPath := strings.TrimLeft(filePath, "/")
	for folder, info := range folders {
		if len(folders) == 1 {
			fullPath = path.Join(folder, fullPath)
		}
		if strings.HasPrefix(fullPath, folder) {
			var u *url.URL
			u, err = url.Parse(info.Repository)
			if err != nil {
				return
			}
			host := u.Host

			switch host {
			case "github.yandex-team.ru", "git.adfox.ru", "github.com":
				u.Scheme = "https"
				u.User = nil
				u.Path = strings.TrimSuffix(u.Path, ".git")
				u.Path = path.Join(u.Path, "blob", info.Revision, info.Path, strings.SplitN(fullPath, "/", 2)[1])
				u.Fragment = fmt.Sprintf("L%d", lineNumber)

				fileURL = u.String()
			case "bb.yandex-team.ru", "bitbucket.sdc.yandex-team.ru":
				u.Path = strings.TrimSuffix(u.Path, ".git")
				parts := strings.Split(strings.TrimLeft(u.Path, "/"), "/")
				project := parts[0]
				repo := parts[1]
				bbURL := url.URL{}
				bbURL.Scheme = "https"
				bbURL.Host = host
				bbURL.Path = path.Join("projects", project, "repos", repo, "browse", info.Path, strings.SplitN(fullPath, "/", 2)[1])
				bbURL.RawQuery = fmt.Sprintf("at=%s", info.Revision)
				bbURL.Fragment = fmt.Sprintf("%d", lineNumber)

				fileURL = bbURL.String()
			case "arcadia.yandex.ru":
				if strings.HasPrefix(fullPath, "src/a.yandex-team.ru/") {
					filePath = strings.SplitN(fullPath, "/", 3)[2]
					fileURL = fmt.Sprintf("https://a.yandex-team.ru/arc_vcs/%s?rev=%s#L%d", filePath, info.Revision, lineNumber)
				} else {
					filePath = strings.SplitN(fullPath, "/", 2)[1]
					fileURL = fmt.Sprintf("https://a.yandex-team.ru%s/%s?rev=%s#L%d", u.Path, filePath, info.Revision, lineNumber)
				}
			case "git.edadeal.yandex-team.ru":
				u.Scheme = "https"
				u.User = nil
				u.Host = "gitlab.edadeal.yandex-team.ru"
				u.Path = strings.TrimSuffix(u.Path, ".git")
				u.Path = path.Join(u.Path, "blob", info.Revision, info.Path, strings.SplitN(fullPath, "/", 2)[1])
				u.Fragment = fmt.Sprintf("L%d", lineNumber)

				fileURL = u.String()
			default:
				err = xerrors.New("repository not supported")
			}

			return
		}
	}

	err = xerrors.New("could not match file to repository")
	return
}
