package checkout

import (
	"bytes"
	"os"
	"os/exec"
	"path"
	"regexp"
	"strings"
	"syscall"

	"a.yandex-team.ru/library/go/core/xerrors"
)

func getGitRevision(path string) (string, error) {
	cmd := exec.Command("git", "rev-parse", "HEAD")
	cmd.Dir = path
	var stdout bytes.Buffer
	cmd.Stdout = &stdout
	cmd.Stderr = os.Stderr
	if err := cmd.Run(); err != nil {
		return "", err
	}

	return strings.TrimRight(stdout.String(), "\n"), nil
}

func GitClone(repoURL, branch, dest string) (folder string, rev string, err error) {
	repoSplit := strings.Split(repoURL, "/")
	folder = repoSplit[len(repoSplit)-1]
	var cmd *exec.Cmd
	if branch != "" {
		cmd = exec.Command("git", "clone", "--depth", "1", "--branch", branch, repoURL, path.Join(dest, folder))
	} else {
		cmd = exec.Command("git", "clone", "--depth", "1", repoURL, path.Join(dest, folder))
	}
	cmd.Stdout = os.Stdout
	cmd.Stderr = os.Stderr
	if err = cmd.Start(); err != nil {
		return
	}

	if err = cmd.Wait(); err != nil {
		if exiterr, ok := err.(*exec.ExitError); ok {
			if status, ok := exiterr.Sys().(syscall.WaitStatus); ok {
				err = xerrors.Errorf("git clone exited with non zero status \"%v\"", status.ExitStatus())
				return
			}
		} else {
			return
		}
	}

	rev, _ = getGitRevision(path.Join(dest, folder))

	return
}

func getYa(dest string) error {
	cmd := exec.Command("svn", "export", "svn+ssh://arcadia.yandex.ru/arc/trunk/arcadia/ya")
	cmd.Dir = dest
	cmd.Stdout = os.Stdout
	cmd.Stderr = os.Stderr
	if err := cmd.Run(); err != nil {
		return err
	}

	cmd = exec.Command("python", "ya", "clone")
	cmd.Dir = dest
	cmd.Stdout = os.Stdout
	cmd.Stderr = os.Stderr
	if err := cmd.Run(); err != nil {
		return err
	}

	return nil
}

func getSvnRevision(path string) (string, error) {
	cmd := exec.Command("svn", "info")
	cmd.Dir = path
	var stdout bytes.Buffer
	cmd.Stdout = &stdout
	cmd.Stderr = os.Stderr
	if err := cmd.Run(); err != nil {
		return "", err
	}

	re, err := regexp.Compile(`Revision: (\d+)`)
	if err != nil {
		return "", err
	}

	revision := re.FindString(stdout.String())

	return strings.Split(revision, " ")[1], nil
}

func ArcadiaClone(repoURL, dest string) (folder string, rev string, err error) {
	repoSplit := strings.Split(strings.TrimRight(repoURL, "/"), "/")
	if len(repoSplit) == 0 {
		err = xerrors.New("incorrect arcadia repository url")
		return
	}

	cmd := exec.Command("svn", "co", repoURL)
	cmd.Dir = dest
	cmd.Stdout = os.Stdout
	cmd.Stderr = os.Stderr

	if err = cmd.Start(); err != nil {
		return
	}

	if err = cmd.Wait(); err != nil {
		if exiterr, ok := err.(*exec.ExitError); ok {
			if status, ok := exiterr.Sys().(syscall.WaitStatus); ok {
				err = xerrors.Errorf("ya make exited with non zero status \"%v\"", status.ExitStatus())
				return
			}
		} else {
			return
		}
	}

	folder = repoSplit[len(repoSplit)-1]
	rev, _ = getSvnRevision(path.Join(dest, folder))

	return
}
