package compress

import (
	"archive/zip"
	"io"
	"os"
	"path/filepath"
	"strings"
)

func ZipFilesToFile(paths []string, prefix string, targetFile *os.File, ignoreDirNames []string) error {
	archive := zip.NewWriter(targetFile)
	defer func() { _ = archive.Close() }()

	for _, sourcePath := range paths {
		err := filepath.Walk(sourcePath, func(path string, info os.FileInfo, err error) error {
			if err != nil {
				return err
			}

			header, err := zip.FileInfoHeader(info)
			if err != nil {
				return err
			}

			if len(paths) > 1 {
				header.Name = strings.TrimPrefix(path, prefix)
			} else {
				header.Name = strings.TrimPrefix(path, sourcePath)
			}

			if info.IsDir() {
				for _, ignoreDirName := range ignoreDirNames {
					if info.Name() == ignoreDirName {
						return filepath.SkipDir
					}
				}
				header.Name += "/"
			} else {
				header.Method = zip.Deflate
			}

			writer, err := archive.CreateHeader(header)
			if err != nil {
				return err
			}

			if !info.Mode().IsRegular() {
				return nil
			}

			file, err := os.Open(path)
			if err != nil {
				return err
			}
			defer func() { _ = file.Close() }()
			_, err = io.Copy(writer, file)

			return err
		})
		if err != nil {
			return err
		}
	}

	return nil
}

func ZipFilesToFilename(source, target string, ignoreDirNames []string) error {
	zipFile, err := os.Create(target)
	if err != nil {
		return err
	}
	defer func() { _ = zipFile.Close() }()

	return ZipFilesToFile([]string{source}, "", zipFile, ignoreDirNames)
}

func UnZipFiles(archive, target string) error {
	reader, err := zip.OpenReader(archive)
	if err != nil {
		return err
	}

	if err := os.MkdirAll(target, 0755); err != nil {
		return err
	}

	for _, file := range reader.File {
		path := filepath.Join(target, file.Name)
		if file.FileInfo().IsDir() {
			_ = os.MkdirAll(path, file.Mode())
			continue
		}

		fileReader, err := file.Open()
		if err != nil {
			return err
		}
		defer func() { _ = fileReader.Close() }()

		targetFile, err := os.OpenFile(path, os.O_WRONLY|os.O_CREATE|os.O_TRUNC, file.Mode())
		if err != nil {
			return err
		}
		defer func() { _ = targetFile.Close() }()

		if _, err := io.Copy(targetFile, fileReader); err != nil {
			return err
		}
	}

	return nil
}
