package nirvana

import (
	"encoding/json"
	"flag"
	"fmt"
	"io/ioutil"
	"os"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/core/log/zap"
	"a.yandex-team.ru/library/go/yandex/tvm"
	tvm_api "a.yandex-team.ru/library/go/yandex/tvm/tvmauth"
	"a.yandex-team.ru/security/impulse/models"
	"a.yandex-team.ru/security/impulse/workflow/internal/api"
	"a.yandex-team.ru/security/impulse/workflow/internal/checkout"
	"a.yandex-team.ru/security/impulse/workflow/internal/config"
	"a.yandex-team.ru/security/impulse/workflow/internal/scan"
)

const (
	// impulse nirvana
	SelfTvmID tvm.ClientID = 2016387
	// impulse prod
	ImpulseTvmID tvm.ClientID = 2016355
)

func StaticEntrypoint(scanner scan.Scan) {
	var err error

	taskIDPtr := flag.String("id", "", "task id")
	pathPtr := flag.String("path", "", "path to repository to check")
	metaPtr := flag.String("meta", "", "checkout meta information")
	s3AccessKeyPtr := flag.String("s3-access-key", "", "s3 secret access key")

	tvmToken := flag.String("tvm-token", "", "imPulse TVM secret")
	oauthToken := flag.String("impulse-oauth-token", "", "imPulse OAuth token")

	sqsOAuthToken := flag.String("sqs-oauth-token", "", "sqs oauth token")
	versionPtr := flag.Bool("version", false, "show current version")
	flag.Parse()

	if *versionPtr {
		fmt.Println(scanner.Version())
		return
	}

	logger, err := zap.New(zap.ConsoleConfig(log.InfoLevel))
	if err != nil {
		fmt.Printf("Failed to initialize logger: %+v", err)
		return
	}

	if *tvmToken == "" && *oauthToken == "" {
		logger.Error("imPulse TVM secret and OAuth token are empty, use one of them")
		return
	}

	var tvmClient *tvm_api.Client
	if *tvmToken != "" {
		tvmSettings := tvm_api.TvmAPISettings{
			SelfID: SelfTvmID,
			ServiceTicketOptions: tvm_api.NewIDsOptions(
				*tvmToken,
				[]tvm.ClientID{ImpulseTvmID}),
			TVMHost: "https://tvm-api.yandex.net",
			TVMPort: 443,
		}

		tvmClient, err = tvm_api.NewAPIClient(tvmSettings, logger)
		if err != nil {
			logger.Errorf("Could not create tvm client: %+v", err)
			return
		}
	}

	impulseAPIOpts := api.Options{
		Endpoint:   "https://impulse.sec.yandex-team.ru/",
		TvmID:      ImpulseTvmID,
		OAuthToken: *oauthToken,
	}
	impulseAPI := api.New(tvmClient, &impulseAPIOpts)

	cfg := config.Config{
		S3Endpoint:        os.Getenv("S3_ENDPOINT"),
		S3Region:          os.Getenv("AWS_REGION"),
		S3AccessKeyID:     os.Getenv("S3_ACCESS_KEY_ID"),
		S3SecretAccessKey: *s3AccessKeyPtr,
		SqsEndpoint:       "https://sqs.yandex.net",
		SqsAccount:        os.Getenv("SQS_ACCOUNT"),
		SqsRetries:        5,
		SqsOAuthToken:     *sqsOAuthToken,
	}

	jsonFile, err := os.Open(*metaPtr)
	defer func() { _ = jsonFile.Close() }()
	if err != nil {
		logger.Errorf("Could not open meta information file: %s", err)
		if err := impulseAPI.UpdateStatus(*taskIDPtr, models.Failed); err != nil {
			logger.Errorf("Could not open udate task status: %s", err)
		}
		return
	}

	byteValue, err := ioutil.ReadAll(jsonFile)
	if err != nil {
		logger.Errorf("Could not read meta information file: %s", err)
		if err := impulseAPI.UpdateStatus(*taskIDPtr, models.Failed); err != nil {
			logger.Errorf("Could not open udate task status: %s", err)
		}
		return
	}

	var meta checkout.Checkout
	err = json.Unmarshal(byteValue, &meta)
	if err != nil {
		logger.Errorf("Could not unmarshal meta information file: %s", err)
		if err := impulseAPI.UpdateStatus(*taskIDPtr, models.Failed); err != nil {
			logger.Errorf("Could not open udate task status: %s", err)
		}
		return
	}

	workflow := NewStaticWorkflow(cfg, *taskIDPtr, *pathPtr, &meta, scanner)
	err = workflow.Init()
	if err != nil {
		logger.Errorf("Could not init static workflow: %s", err)
		if err := impulseAPI.UpdateStatus(*taskIDPtr, models.Failed); err != nil {
			logger.Errorf("Could not open udate task status: %s", err)
		}
		return
	}
	_, err = workflow.Run()
	if err != nil {
		logger.Errorf("Static workflow finished with error: %s", err)
		if err := impulseAPI.UpdateStatus(*taskIDPtr, models.Failed); err != nil {
			logger.Errorf("Could not open udate task status: %s", err)
		}
		return
	}
	_ = workflow.Done()
}
