package nirvana

import (
	"encoding/json"
	"fmt"
	"log"
	"time"

	"a.yandex-team.ru/security/impulse/models"
	"a.yandex-team.ru/security/impulse/pkg/queue"
	"a.yandex-team.ru/security/impulse/pkg/s3"
	"a.yandex-team.ru/security/impulse/workflow/internal/checkout"
	"a.yandex-team.ru/security/impulse/workflow/internal/config"
	"a.yandex-team.ru/security/impulse/workflow/internal/scan"
)

const (
	RawReportsBucket = "rawreports"
)

type StaticWorkflow struct {
	Workflow
	S3           s3.S3Uploader
	Queue        queue.Queue
	SourcePath   string
	CheckoutMeta *checkout.Checkout
	Scanner      scan.Scan
}

func NewStaticWorkflow(cfg config.Config, taskID, sourcePath string, meta *checkout.Checkout, scanner scan.Scan) *StaticWorkflow {
	return &StaticWorkflow{
		Workflow: Workflow{
			TaskID: taskID,
			CFG:    cfg,
		},
		CheckoutMeta: meta,
		SourcePath:   sourcePath,
		Scanner:      scanner,
	}
}

func (n *StaticWorkflow) Init() (err error) {
	if n.CFG.S3SecretAccessKey != "" {
		n.S3, err = s3.NewUploader(n.CFG.S3Config())
		if err != nil {
			return
		}
	}
	n.Queue, err = queue.New(n.CFG.QueueConfig())
	if err != nil {
		return
	}
	return
}

func (n *StaticWorkflow) Run() (string, error) {
	log.SetPrefix("[" + string(n.Scanner.Type()) + "] ")
	defer log.SetPrefix("")

	startTime := time.Now().Unix()
	log.Println("Start processing")

	err := n.Scanner.Init(n.SourcePath, n.CheckoutMeta)
	if err != nil {
		log.Println("Could not init analyzer:", err)
		return "", err
	}
	log.Println("Launching analyzer")
	err = n.Scanner.Run()
	if err != nil {
		log.Println("Could not launch analyzer:", err)
		return "", err
	}
	log.Println("Analyzer successfully completed")

	rawReportURL := ""
	if n.S3 != nil {
		log.Println("Start uploading raw report to s3")
		rawReport := n.Scanner.RawReport()
		rawReportKey := fmt.Sprintf("%s_%s", n.TaskID, n.Scanner.Type())
		rawReportURL, err := n.S3.S3UploadString(RawReportsBucket, rawReportKey, rawReport)
		if err != nil {
			log.Println("Could not upload to S3:", err)
			return "", err
		}
		log.Println("Successfully uploaded raw report to s3:", rawReportURL)
	} else {
		log.Println("Skipping uploading raw report to s3, empty s3-access-key")
	}

	log.Println("Start normalization process")
	err = n.Scanner.Normalize()
	if err != nil {
		log.Println("Could not finish normalization process:", err)
		return "", err
	}
	log.Println("Normalization process successfully completed")

	endTime := time.Now().Unix()
	revision := ""
	for _, folder := range n.CheckoutMeta.Folders {
		if folder.IsArcadia {
			revision = folder.Revision
			break
		}
	}
	launchResult := models.Result{
		ResultMeta: models.ResultMeta{
			TaskID:       n.TaskID,
			Type:         n.Scanner.Type(),
			RawReportURL: rawReportURL,
			CommitHash:   revision,
			StartTime:    startTime,
			EndTime:      endTime,
		},
		Report: n.Scanner.Report(),
	}

	data, _ := json.MarshalIndent(launchResult, "", "    ")

	normalizedReportURL := ""
	if n.S3 != nil {
		log.Println("Start uploading normalized report to s3")
		rawReportKey := fmt.Sprintf("normalized_%s_%s", n.TaskID, n.Scanner.Type())
		normalizedReportURL, err = n.S3.S3UploadString(RawReportsBucket, rawReportKey, string(data))
		if err != nil {
			log.Println("Could not upload to S3:", err)
			return "", err
		}
		log.Println("Successfully uploaded normalized report to s3:", normalizedReportURL)
	} else {
		log.Println("Skipping uploading normalized report to s3, empty s3-access-key")
	}

	body, _ := json.Marshal(models.ReportLoadRequest{
		NormalizedReportURL: normalizedReportURL,
		Type:                n.Scanner.Type(),
	})

	if _, err := n.Queue.SendMessage(&queue.SendOptions{
		QueueURL: n.CFG.ReportsQueueURL(),
		Msg:      string(body),
	}); err != nil {
		log.Println("Could not send message:", err)
		return "", err
	}

	log.Println("Finish processing")

	return string(data), nil
}

func (n *StaticWorkflow) Done() (err error) {

	return
}
