package main

import (
	"bytes"
	"encoding/json"
	"log"
	"os"
	"os/exec"
	"path"
	"strings"
	"syscall"

	"a.yandex-team.ru/security/impulse/models"
	"a.yandex-team.ru/security/impulse/workflow/internal/checkout"
	"a.yandex-team.ru/security/impulse/workflow/internal/dedup"
)

type BanditVuln struct {
	Code            string `json:"code"`
	Filename        string `json:"filename"`
	IssueConfidence string `json:"issue_confidence"`
	IssueSeverity   string `json:"issue_severity"`
	IssueText       string `json:"issue_text"`
	LineNumber      int    `json:"line_number"`
	LineRange       []int  `json:"line_range"`
	MoreInfo        string `json:"more_info"`
	TestID          string `json:"test_id"`
	TestName        string `json:"test_name"`
}

type BanditReport struct {
	Results []BanditVuln `json:"results"`
}

type Bandit struct {
	python       *string
	basePath     string
	checkoutMeta *checkout.Checkout
	rawReport    map[string][]byte
	report       []*models.NewVulnerabilityDeduplicationRequestDTO
}

var severityMap = map[string]models.SeverityType{
	"LOW":    models.Low,
	"MEDIUM": models.Medium,
	"HIGH":   models.Critical,
}

func (r *Bandit) Init(sourcePath string, checkoutMeta *checkout.Checkout) error {
	r.basePath = sourcePath
	r.checkoutMeta = checkoutMeta
	r.rawReport = make(map[string][]byte)

	return nil
}

func (r *Bandit) Run() error {
	interpreter := "python" + *r.python
	for folder := range r.checkoutMeta.Folders {
		sourcePath := path.Join(r.basePath, folder)
		cmd := exec.Command(interpreter, "/usr/local/bin/bandit", "-f", "json", "-n", "10", "-s", "B101", "-r", sourcePath)
		var stdout bytes.Buffer
		cmd.Stdout = &stdout
		cmd.Stderr = os.Stderr

		r.rawReport[folder] = nil

		if err := cmd.Run(); err != nil {
			if exiterr, ok := err.(*exec.ExitError); ok {
				if status, ok := exiterr.Sys().(syscall.WaitStatus); ok {
					exitStatus := status.ExitStatus()
					if exitStatus != 1 {
						log.Printf("Unexpected exit status \"%v\" for %v\n", exitStatus, cmd.Args)
					} else {
						r.rawReport[folder] = stdout.Bytes()
					}
				}
			}
		} else {
			r.rawReport[folder] = stdout.Bytes()
		}
	}
	return nil
}

func (r *Bandit) Normalize() error {
	if len(r.rawReport) == 0 {
		r.report = []*models.NewVulnerabilityDeduplicationRequestDTO{}
		return nil
	}

	r.report = make([]*models.NewVulnerabilityDeduplicationRequestDTO, 0)

	for folder, rawReport := range r.rawReport {
		report := BanditReport{}
		err := json.Unmarshal(rawReport, &report)
		if err != nil {
			log.Println("Could not unmarshal bandit report:", err)
			continue
		}

		for _, banditVuln := range report.Results {
			filePath := strings.TrimPrefix(banditVuln.Filename, path.Join(r.basePath, folder))
			hash, _ := dedup.GetLineHash(banditVuln.Filename, banditVuln.LineNumber)
			fileURL, _ := checkout.GenerateFileURLWithLineNumber(r.checkoutMeta.Folders[folder], filePath, banditVuln.LineNumber)

			keyProps := models.VulnerabilityProperties{
				"filename":    filePath,
				"rule":        banditVuln.TestID,
				"line_number": banditVuln.LineNumber,
				"line_hash":   hash,
				"description": banditVuln.IssueText,
			}
			displayProps := models.VulnerabilityProperties{
				"code":     banditVuln.Code,
				"severity": severityMap[banditVuln.IssueSeverity],
				"file_url": fileURL,
			}

			r.report = append(r.report, &models.NewVulnerabilityDeduplicationRequestDTO{
				Severity:          severityMap[banditVuln.IssueSeverity],
				Category:          banditVuln.TestName,
				KeyProperties:     keyProps,
				DisplayProperties: displayProps,
			})
		}
	}

	return nil
}

func (r *Bandit) Type() models.ScanTypeName {
	switch *r.python {
	case "2":
		return models.BANDIT2
	case "3":
		return models.BANDIT3
	default:
		return ""
	}
}

func (r *Bandit) RawReport() string {
	rawReport := "[\n"
	for _, rawSingleReport := range r.rawReport {
		rawReport += strings.TrimRight(string(rawSingleReport), "\n") + ",\n"
	}
	if len(rawReport) > 4 {
		rawReport = rawReport[:len(rawReport)-2]
	}
	rawReport += "]"

	return rawReport
}

func (r *Bandit) Report() interface{} {
	return r.report
}

func (r *Bandit) Version() string {
	return "0.1"
}
