package main

import (
	"log"
	"os"
	"os/exec"
	"path"
	"strings"
)

var supportedLanguages = map[string]struct {
	Language string
}{
	"java": {"java"},
	"go":   {"go"},
	"js":   {"js"},
}

func prepareLanguages(langsPtr *string) []string {
	languages := []string{}

	if langsPtr == nil || len(*langsPtr) == 0 {
		for language := range supportedLanguages {
			languages = append(languages, language)
		}
		return languages
	}

	specifiedLanguages := strings.Split(*langsPtr, ",")
	for _, language := range specifiedLanguages {
		if _, ok := supportedLanguages[language]; ok {
			languages = append(languages, language)
		}
	}
	return languages
}

// Side Component Analysis
func (r *Blackduck) scaGoConditions(sourcePath string) bool {
	//
	check := exec.Command("ls", "-l", sourcePath)
	check.Stdout = os.Stdout
	check.Stderr = os.Stderr
	_ = check.Run()
	//
	var checkfile string
	checkfile = path.Join(sourcePath, "Gopkg.lock")
	if _, err := os.Stat(checkfile); os.IsNotExist(err) {
		log.Println(sourcePath, "Gopkg.lock not found...")
		checkfile = path.Join(sourcePath, "vendor/vendor.json")
		if _, err := os.Stat(checkfile); os.IsNotExist(err) {
			log.Println(sourcePath, "vendor/vendor.json not found...")
			checkfile = path.Join(sourcePath, "vendor.conf")
			if _, err := os.Stat(checkfile); os.IsNotExist(err) {
				log.Println(sourcePath, "vendor.json not found...")
				checkfile = path.Join(sourcePath, "go.mod")
				if _, err := os.Stat(checkfile); os.IsNotExist(err) {
					log.Println(sourcePath, "go.mod not found...")
					return false
				}

				cmd := exec.Command("go", "list", "-m")
				cmd.Dir = sourcePath
				cmd.Stdout = os.Stdout
				cmd.Stderr = os.Stderr
				err1 := cmd.Run()
				cmd = exec.Command("go", "mod", "graph")
				cmd.Dir = sourcePath
				cmd.Stdout = os.Stdout
				cmd.Stderr = os.Stderr
				err2 := cmd.Run()
				cmd = exec.Command("go", "mod", "why")
				cmd.Dir = sourcePath
				cmd.Stdout = os.Stdout
				cmd.Stderr = os.Stderr
				err3 := cmd.Run()
				if err1 != nil || err2 != nil || err3 != nil {
					log.Println(sourcePath, "cant solve dependencies", err1, err2, err3)
					return false
				}
			}
		}
	}

	return true
}

func (r *Blackduck) scaJavaConditions(sourcePath string) bool {

	checkfile := path.Join(sourcePath, "pom.xml")
	if _, err := os.Stat(checkfile); os.IsNotExist(err) {
		log.Println(sourcePath, "pom.xml not found...")
		return false
	}

	cmd := exec.Command("mvn", "-am", "-pl", sourcePath, "dependency:tree")
	cmd.Dir = sourcePath
	cmd.Stdout = os.Stdout
	cmd.Stderr = os.Stderr
	if err := cmd.Run(); err != nil {
		log.Println(sourcePath, "mvn: cant build dependency tree")
		return false
	}
	return true
}

func (r *Blackduck) scaJsConditions(sourcePath string) bool {

	cmd := exec.Command("npm", "install", "--package-lock-only")
	cmd.Dir = sourcePath
	if err := cmd.Run(); err != nil {
		log.Println(sourcePath, "npm: cant install")
		return false
	}

	var checkfile string
	checkfile = path.Join(sourcePath, "package.json")
	if _, err := os.Stat(checkfile); os.IsNotExist(err) {
		log.Println(sourcePath, "package.json not found...")
		return false
	}
	checkfile = path.Join(sourcePath, "package-lock.json")
	if _, err := os.Stat(checkfile); os.IsNotExist(err) {
		log.Println(sourcePath, "package-lock.json not found...")
		return false
	}

	return true
}

func (m *blackduckMeta) getFromMeta(key string) string {
	for _, link := range m.Links {
		if link.Rel == key {
			return link.Href
		}
	}
	return "null"
}
