package main

import (
	"encoding/json"
	"fmt"
	"log"
	"os"
	"os/exec"
	"path"
	"strings"
	"syscall"

	"github.com/go-resty/resty/v2"
	"github.com/gofrs/uuid"

	"a.yandex-team.ru/security/impulse/models"
	"a.yandex-team.ru/security/impulse/workflow/internal/checkout"
)

// supported langs
const (
	GO   string = "go"
	JAVA string = "java"
	JS   string = "js"
	//BANDIT      ScanTypeName = "bandit_scan"
	//BANDIT2     ScanTypeName = "bandit2_scan"
)

// supported detectors
const (
	GOMOD    string = "GO_MOD"
	GODEP    string = "GO_DEP"
	GOVNDR   string = "GO_VNDR"
	GOVENDOR string = "GO_VENDOR"
	MAVEN    string = "MAVEN"
	NPM      string = "NPM"
)

type Blackduck struct {
	basePath     string
	checkoutMeta *checkout.Checkout
	rawReport    map[string]vulnsRepresentation
	report       []*models.NewVulnerabilityDeduplicationRequestDTO
	// scanner specific stuff
	langsArg          *string
	languages         []string
	apiToken          *string
	auth              authInfo
	synopsysDetectJar *string
	httpc             *resty.Client
}

func (r *Blackduck) Init(sourcePath string, checkoutMeta *checkout.Checkout) error {

	// general
	r.basePath = sourcePath
	_ = os.Setenv("GOPATH", sourcePath)
	r.checkoutMeta = checkoutMeta
	r.rawReport = make(map[string]vulnsRepresentation) // blackduck report format

	r.languages = prepareLanguages(r.langsArg)

	r.httpc = resty.New().
		SetHeader("User-Agent", "impulse-codeql <security@yandex-team.ru>")

	return nil
}

func (r *Blackduck) Run() error {
	// claim bearer token
	resp, err := r.httpc.R().
		SetHeader("Accept", "application/vnd.blackducksoftware.user-4+json").
		SetHeader("Authorization", fmt.Sprintf("token %s", *r.apiToken)).
		Post(fmt.Sprintf("%s%s", BlackduckURL, "/api/tokens/authenticate"))
	if err != nil {
		return fmt.Errorf("can`t get bearer token at BlackDuck server: %v", err)
	}

	err = json.Unmarshal(resp.Body(), &r.auth)
	if err != nil {
		return fmt.Errorf("could not unmarshal response for auth: %v", err)
	}

	for folder := range r.checkoutMeta.Folders {
		sourcePath := path.Join(r.basePath, folder)
		relSourcePath := r.checkoutMeta.Folders[folder].Path

		_, err := os.Stat(sourcePath)
		if os.IsNotExist(err) {
			log.Printf("File or directory %s does not exists: %v\n", sourcePath, err)
			continue
		}

		// additional preparation
		var cmd *exec.Cmd
		cmd = exec.Command("git", "submodule", "update", "--init", "--recursive")
		cmd.Dir = sourcePath
		cmd.Stdout = os.Stdout
		cmd.Stderr = os.Stderr
		if err := cmd.Run(); err != nil {
			log.Printf("Can`t execute 'git submodule' command: %s\n", err.Error())
			return err
		}

		// check prerequisites (we could add detectors per succeeded check and stop scanning only if nothing been met)
		var detectIncludedDetectorTypes []string
		for _, lang := range r.languages {
			switch lang {
			case GO:
				//if !r.scaGoConditions(sourcePath) {
				//	return errors.New("'go' prerequisites not met")
				//}
				detectIncludedDetectorTypes = append(detectIncludedDetectorTypes, GOMOD, GODEP, GOVNDR, GOVENDOR)
			case JAVA:
				//if !r.scaJavaConditions(sourcePath) {
				//	return errors.New("'java' prerequisites not met")
				//}
				detectIncludedDetectorTypes = append(detectIncludedDetectorTypes, MAVEN)
			case JS:
				//if !r.scaJsConditions(sourcePath) {
				//	return errors.New("'js' prerequisites not met")
				//}
				detectIncludedDetectorTypes = append(detectIncludedDetectorTypes, NPM)
			}
		}

		// run scan
		taskID := uuid.Must(uuid.NewV4()).String()
		///
		log.Println(strings.Join(detectIncludedDetectorTypes, ","))
		log.Printf("taskID: %s\n", taskID)
		///
		// unbind scanning from report collecting. make two for loops. dodge awaiting.
		log.Println(r.basePath, folder, r.checkoutMeta.Path, r.checkoutMeta.Folders[folder].Repository, r.checkoutMeta.Folders[folder].Path)
		//
		cmd = exec.Command("java", "-jar", *r.synopsysDetectJar,
			fmt.Sprintf("--blackduck.url=%s", BlackduckURL), fmt.Sprintf("--blackduck.api.token=%s", *r.apiToken),
			fmt.Sprintf("--detect.source.path=%s", strings.TrimSuffix(sourcePath, relSourcePath)), fmt.Sprintf("--detect.required.detector.types=%s", strings.Join(detectIncludedDetectorTypes, ",")), // strings.Join(detectIncludedDetectorTypes, ",")
			"--detect.project.name=test", "--detect.tools=DETECTOR",
			fmt.Sprintf("--detect.project.version.name=%s", taskID),
			fmt.Sprintf("--detect.maven.build.command=-am -pl %s", relSourcePath), "--detect.parallel.processors=0", "--detect.impact.analysis.enabled=true",
			"--detect.wait.for.results=true", "--detect.timeout=6000",
		)
		//cmd.Dir = strings.TrimSuffix(sourcePath, relSourcePath)
		cmd.Stdout = os.Stdout
		cmd.Stderr = os.Stderr
		if err := cmd.Run(); err != nil {
			if exiterr, ok := err.(*exec.ExitError); ok {
				if status, ok := exiterr.Sys().(syscall.WaitStatus); ok {
					exitStatus := status.ExitStatus()
					if exitStatus != 0 {
						return fmt.Errorf("unexpected exit status \"%v\" for %v", exitStatus, cmd.Args)
					}
				}
			}
		}

		// split loop HERE

		// gather report
		resp, err := r.httpc.R().
			SetHeader("Accept", "application/vnd.blackducksoftware.project-detail-4+json").
			SetHeader("Authorization", fmt.Sprintf("bearer %s", r.auth.BearerToken)).
			SetQueryParam("q", fmt.Sprintf("name:%s", "test")).
			Get(fmt.Sprintf("%s%s", BlackduckURL, "/api/projects"))
		if err != nil {
			return fmt.Errorf("can`t get project from BlackDuck server: err: %v", err)
		}

		var projects projectRepresentations
		err = json.Unmarshal(resp.Body(), &projects)
		if err != nil {
			return fmt.Errorf("could not unmarshal response for projects list: %v", err)
		}

		projNum := -1
		for i, proj := range projects.Items {
			if proj.Name == "test" {
				projNum = i
			}
		}
		if projNum == -1 {
			return fmt.Errorf("project 'test' doesnt exist")
		}

		resp, err = r.httpc.R().
			SetHeader("Accept", "application/vnd.blackducksoftware.project-detail-5+json").
			SetHeader("Authorization", fmt.Sprintf("bearer %s", r.auth.BearerToken)).
			SetQueryParam("q", fmt.Sprintf("versionName:%s", taskID)).
			Get(projects.Items[projNum].Meta.getFromMeta("versions"))
		if err != nil {
			return fmt.Errorf("can`t get version from BlackDuck server: err: %v", err)
		}

		var version projectVersionRepresentations
		err = json.Unmarshal(resp.Body(), &version)
		if err != nil {
			return fmt.Errorf("could not unmarshal response for versions list: %v", err)
		}
		if version.TotalCount != 1 || version.Items[0].VersionName != taskID {
			return fmt.Errorf("version ambiguity on blackduck server")
		}

		resp, err = r.httpc.R().
			SetHeader("Accept", "application/vnd.blackducksoftware.bill-of-materials-6+json").
			SetHeader("Authorization", fmt.Sprintf("bearer %s", r.auth.BearerToken)).
			SetQueryParam("limit", "1000").
			Get(version.Items[0].Meta.getFromMeta("vulnerable-components"))
		if err != nil {
			return fmt.Errorf("can`t get vulns from BlackDuck server: err: %v", err)
		}

		var vulns vulnsRepresentation
		err = json.Unmarshal(resp.Body(), &vulns)
		if err != nil {
			return fmt.Errorf("could not unmarshal response for vulns: %v", err)
		}

		r.rawReport[folder] = vulns
	}

	return nil
}

func (r *Blackduck) Normalize() error {
	if len(r.rawReport) == 0 {
		r.report = []*models.NewVulnerabilityDeduplicationRequestDTO{}
		return nil
	}

	r.report = make([]*models.NewVulnerabilityDeduplicationRequestDTO, 0)

	for folder, report := range r.rawReport {
		for _, vuln := range report.Items {
			keyProperties := models.VulnerabilityProperties{
				"package_name": vuln.ComponentName,
				"id":           vuln.ComponentVersionOriginID,
				"version":      vuln.ComponentVersionName,
				"is_arcadia":   r.checkoutMeta.Folders[folder].IsArcadia,
				"folder":       folder,
			}
			fileURL, _ := checkout.GenerateFileURL(r.checkoutMeta.Folders[folder], "")
			displayProperties := models.VulnerabilityProperties{
				"vulnerable_versions": vuln.ComponentVersionName,
				"vulnerability_name":  vuln.VulnerabilityWithRemediation.VulnerabilityName,
				"description":         vuln.VulnerabilityWithRemediation.Description,
				"reference":           vuln.VulnerabilityWithRemediation.RelatedVulnerability,
				"cvss_score":          vuln.VulnerabilityWithRemediation.BaseScore,
				"file_url":            fileURL,
			}
			issue := &models.NewVulnerabilityDeduplicationRequestDTO{
				Severity:          models.SeverityType(vuln.VulnerabilityWithRemediation.Severity),
				Category:          "Vulnerable Component",
				KeyProperties:     keyProperties,
				DisplayProperties: displayProperties,
			}
			r.report = append(r.report, issue)
		}
	}

	return nil
}

func (r *Blackduck) Type() models.ScanTypeName {
	return models.BLACKDUCK
}

func (r *Blackduck) RawReport() string {
	rawReport, err := json.Marshal(r.rawReport)
	if err != nil {
		log.Printf("Unable to marshal raw report: %v", err)
		return ""
	}
	return string(rawReport)
}

func (r *Blackduck) Report() interface{} {
	return r.report
}

func (r *Blackduck) Version() string {
	return "0.1"
}
