package main

import (
	"encoding/json"
	"flag"
	"fmt"
	"io/ioutil"
	"log"
	"net/url"
	"os"

	"a.yandex-team.ru/library/go/core/xerrors"
	"a.yandex-team.ru/security/impulse/models"
	"a.yandex-team.ru/security/impulse/workflow/internal/checkout"
	"a.yandex-team.ru/security/impulse/workflow/internal/compress"
)

func clone(repoURL, branch, path, sshKeyPath, username string) (folder string, rev string, err error) {
	u, err := url.Parse(repoURL)
	if err != nil {
		return
	}

	if sshKeyPath != "" {
		currentUsername := ""
		if u.User.Username() != "" {
			currentUsername = u.User.Username()
		} else if username != "" {
			currentUsername = username
		}
		if currentUsername != "" {
			err = setSSHKey(sshKeyPath, currentUsername)
			if err != nil {
				log.Println("Could not set ssh private key:", err)
				return
			}
		}
	}

	switch u.Host {
	case "github.yandex-team.ru", "bb.yandex-team.ru", "git.adfox.ru", "gitlab.edadeal.yandex-team.ru":
		folder, rev, err = checkout.GitClone(repoURL, branch, path)
	case "arcadia.yandex.ru":
		folder, rev, err = checkout.ArcadiaClone(repoURL, path)
	default:
		err = xerrors.New("repository not supported")
	}

	if err != nil {
		log.Printf("Failed to clone repo %s: %v\n", repoURL, err)
		return
	}

	return
}

func setSSHKey(sshKeyPath, user string) (err error) {
	sshCmd := fmt.Sprintf("ssh -v -o StrictHostKeyChecking=no -i %s -l %s", sshKeyPath, user)
	log.Println("Set ssh command:", sshCmd)
	_ = os.Setenv("GIT_SSH_COMMAND", sshCmd)
	_ = os.Setenv("SVN_SSH", sshCmd)
	return
}

func main() {
	var (
		rawRepositories string
		username        string
		sshKeyPath      string
		metaPath        string
		outputPath      string
	)
	flag.StringVar(&rawRepositories, "repositories", "", "list of repositories")
	flag.StringVar(&username, "username", "", "gh/bb username")
	flag.StringVar(&sshKeyPath, "ssh-key", "", "gh/bb ssh private key")
	flag.StringVar(&metaPath, "meta", "", "checkout meta information")
	flag.StringVar(&outputPath, "output", "", "output file destination")
	flag.Parse()

	log.SetPrefix("[checkout] ")

	dest, err := ioutil.TempDir("/tmp", "repo_")
	if err != nil {
		return
	}
	log.Println("Cloning to", dest)

	var repositories []models.Repository
	err = json.Unmarshal([]byte(rawRepositories), &repositories)
	if err != nil {
		log.Printf("Could not unmarshal repositories json %s: %v\n", rawRepositories, err)
		return
	}

	meta := checkout.Checkout{
		Path:    "",
		Folders: make(map[string]checkout.Folder),
	}

	for _, repository := range repositories {
		log.Printf("Cloning %s:%s to %s\n", repository.URL, repository.Branch, dest)
		folder, rev, err := clone(repository.URL, repository.Branch, dest, sshKeyPath, username)

		if len(repositories) == 1 {
			meta.Path = folder
		}
		meta.Folders[folder] = checkout.Folder{
			Repository: repository.URL,
			Branch:     repository.Branch,
			Revision:   rev,
		}

		if err != nil {
			log.Printf("Could not clone %s:%s: %v\n", repository.URL, repository.Branch, err)
			return
		}
	}
	log.Println("Repositories cloned to", dest)

	err = compress.ZipFilesToFilename(dest, outputPath, []string{".git", ".svn"})
	if err != nil {
		log.Println("Could not compress files:", err)
		return
	}

	jsonFile, _ := os.Create(metaPath)
	defer func() { _ = jsonFile.Close() }()
	data, _ := json.MarshalIndent(meta, "", "    ")
	_, _ = jsonFile.Write(data)

	log.Println("Successfully saved compressed repository to", outputPath)
}
