package main

import (
	"fmt"
	"io/ioutil"
	"net/http"
	"os"
	"path"
	"strings"

	"github.com/go-resty/resty/v2"

	"a.yandex-team.ru/security/impulse/workflow/internal/checkout"
)

type CodeqlEnvironment struct {
	Home            string
	oldHome         string
	httpc           *resty.Client
	queryCacheNames []string
	SearchPath      string
	queriesBranch   string
}

func NewCodeqlEnvironment(oldHome string, httpc *resty.Client, queriesBranch *string) (*CodeqlEnvironment, error) {
	codeqlHome, err := ioutil.TempDir("", "codeql-home")
	if err != nil {
		return nil, fmt.Errorf("codeql environment error. cant create codeql home folder. %v", err)
	}
	searchPath := path.Join(codeqlHome, "codeql-yandex-queries")
	// fmt.Printf("[+] codeql-home folder: %s\n", codeqlHome)

	branch := ""
	if queriesBranch != nil && len(*queriesBranch) > 0 {
		branch = *queriesBranch
	}

	codeqlEnvironment := CodeqlEnvironment{
		Home:            codeqlHome,
		oldHome:         oldHome,
		httpc:           httpc,
		queryCacheNames: []string{"codeql-queries", "codeql-go-queries", "codeql-yandex-queries"},
		SearchPath:      searchPath,
		queriesBranch:   branch,
	}
	return &codeqlEnvironment, nil
}

func (env CodeqlEnvironment) PrepareCodeqlCli() error {

	oldCodeqlCliPath := path.Join(env.oldHome, "codeql-cli")
	if _, err := os.Stat(oldCodeqlCliPath); os.IsNotExist(err) {
		return fmt.Errorf("codeql environment error. codeql-cli at %s not found. %v", oldCodeqlCliPath, err)
	}

	newCodeqlCliPath := path.Join(env.Home, "codeql-cli")
	args := []string{"cp", "-r", oldCodeqlCliPath, newCodeqlCliPath}
	err := execCommand(args)
	if err != nil {
		return fmt.Errorf("codeql environment error. on moving codeql-cli. err: %v", err)
	}

	return nil
}

func (env CodeqlEnvironment) Clean() error {
	return os.RemoveAll(env.Home)
}

func (env CodeqlEnvironment) GetNewestCache(resp1, resp2 *resty.Response) (*resty.Response, error) {
	headers1 := resp1.Header()
	headers2 := resp2.Header()
	time1, err1 := http.ParseTime(headers1.Get("Last-Modified"))
	time2, err2 := http.ParseTime(headers2.Get("Last-Modified"))

	switch {
	case err1 != nil && err2 != nil:
		return nil, fmt.Errorf("codeql environment error. incorrect Last-Modified header for both caches. err1: %v, err2: %v", err1, err2)
	case err1 != nil:
		return resp2, nil
	case err2 != nil:
		return resp1, nil
	default:
		if time1.After(time2) {
			return resp1, nil
		} else {
			return resp2, nil
		}
	}
}

func (env CodeqlEnvironment) Prepare() error {
	if err := env.PrepareCodeqlCli(); err != nil {
		return fmt.Errorf("codeql environment error. on prepare cli environment. err: %v", err)
	}
	if err := env.PrepareQueries(); err != nil {
		return fmt.Errorf("codeql environment error. on prepare queries environment. err: %v", err)
	}
	return nil
}

func (env CodeqlEnvironment) PrepareQueries() error {

	// get cached queries
	if env.queriesBranch == "" {
		for _, name := range env.queryCacheNames {
			if err := env.PrepareSpecificQueries(name); err != nil {
				return fmt.Errorf("codeql environment error. unable to prepare %s cache. err: %v", name, err)
			}
		}
		return nil
	}

	// get queries from bb repo if branch is specified
	folder, _, _ := checkout.GitClone("ssh://git@bb.yandex-team.ru/sec/codeql.git", env.queriesBranch, env.Home)
	args := []string{"ls", "-l", env.Home}
	_ = execCommand(args)
	args = []string{"ls", "-l", path.Join(env.Home, folder)}
	_ = execCommand(args)

	folderGo, _, _ := checkout.GitClone("ssh://git@bb.yandex-team.ru/sec/codeql-go.git", env.queriesBranch, env.Home)
	args = []string{"ls", "-l", env.Home}
	_ = execCommand(args)
	args = []string{"ls", "-l", path.Join(env.Home, folderGo)}
	_ = execCommand(args)

	return nil
}

func (env CodeqlEnvironment) PrepareSpecificQueries(name string) error {
	fmt.Printf("Processing cache: %s", name)
	resp1, err1 := env.httpc.R().Get(fmt.Sprintf("https://codeqlindex.s3.mds.yandex.net/cache1-%s.tar.gz", name))
	resp2, err2 := env.httpc.R().Get(fmt.Sprintf("https://codeqlindex.s3.mds.yandex.net/cache2-%s.tar.gz", name))

	var resp *resty.Response

	switch {
	case err1 != nil && err2 != nil:
		return fmt.Errorf("codeql environment error. unable to fetch %s cache. %v. %v", name, err1, err2)
	case err1 != nil:
		if resp2.IsSuccess() {
			resp = resp2
		} else {
			return fmt.Errorf("codeql environment error. unable to fetch %s cache. resp1: %v. resp2 not succeed", name, err1)
		}
	case err2 != nil:
		if resp1.IsSuccess() {
			resp = resp1
		} else {
			return fmt.Errorf("codeql environment error. unable to fetch %s cache. resp2: %v. resp1 not succeed", name, err2)
		}
	default:
		switch {
		case !resp1.IsSuccess() && !resp2.IsSuccess():
			return fmt.Errorf("codeql environment error. unable to fetch %s cache. resp1 and resp2 not succeed", name)
		case !resp1.IsSuccess():
			resp = resp2
		case !resp2.IsSuccess():
			resp = resp1
		default:
			newestResp, err := env.GetNewestCache(resp1, resp2)
			if err != nil {
				return fmt.Errorf("codeql environment error. unable to get newest %s cache. err: %v", name, err)
			}
			resp = newestResp
		}
	}

	archivePath := path.Join(env.Home, fmt.Sprintf("%s.tar.gz", name))
	f, err := os.Create(archivePath)
	if err != nil {
		return fmt.Errorf("codeql environment error. Unable to create %s. err: %v", name, err)
	}
	_, _ = f.Write(resp.Body())
	_ = f.Close()

	args := []string{"tar", "xzvf", archivePath, "-C", env.Home}
	err = execCommand(args)
	if err != nil {
		return fmt.Errorf("codeql environment error. on tar decompression of %s. err: %v", name, err)
	}

	return nil
}

func (env CodeqlEnvironment) FindQuery(relPath string) (string, bool) {
	for _, name := range env.queryCacheNames {
		fullPath := path.Join(env.Home, name, relPath)
		// fmt.Printf("[+] FindQuery. env.Home: %s\n", env.Home)
		// fmt.Printf("[+] FindQuery. name: %s\n", name)
		// fmt.Printf("[+] FindQuery. relPath: %s\n", relPath)
		// fmt.Printf("[+] FindQuery. fullPath: %s\n", fullPath)
		if !strings.HasPrefix(fullPath, env.Home) {
			return "", false
		}
		if _, err := os.Stat(fullPath); !os.IsNotExist(err) {
			return fullPath, true
		}
	}
	return "", false
}
