package main

import ( // "fmt"
	"os"
	"os/exec"
	"path"
	"strings"
	"syscall"

	"a.yandex-team.ru/library/go/core/xerrors"
)

func prepareLanguages(langsPtr *string) []string {
	languages := []string{}

	if langsPtr == nil || len(*langsPtr) == 0 {
		for language := range SupportedLanguages {
			languages = append(languages, language)
		}
		return languages
	}

	specifiedLanguages := strings.Split(*langsPtr, ",")
	for _, language := range specifiedLanguages {
		if _, ok := SupportedLanguages[language]; ok {
			languages = append(languages, language)
		}
	}
	return languages
}

func prepareQueries(queriesPtr *string, languages []string, codeqlEnv *CodeqlEnvironment) []string {
	querySuites := []string{}
	finalQuerySuitesMap := map[string]bool{}

	if queriesPtr != nil && len(*queriesPtr) > 0 {
		querySuites = strings.Split(*queriesPtr, ",")
	}

	for _, language := range languages {
		found := false
		for _, querySuite := range querySuites {
			if strings.HasPrefix(querySuite, language) && !strings.Contains(querySuite, "/") {
				finalQuerySuitesMap[querySuite] = true
				found = true
			}
			queryPath, ok := codeqlEnv.FindQuery(querySuite)
			// fmt.Printf("[+] FindQuery. querySuite: %s -> %s [%t]", querySuite, queryPath, ok)
			if ok && strings.Contains(queryPath, language) {
				finalQuerySuitesMap[queryPath] = true
				found = true
			}
		}
		if !found {
			if v, ok := SupportedLanguages[language]; ok {
				finalQuerySuitesMap[v.DefaultQuerySuite] = true
			}
		}
	}

	finalQuerySuites := []string{}
	for value := range finalQuerySuitesMap {
		finalQuerySuites = append(finalQuerySuites, value)
	}

	return finalQuerySuites
}

func execCommand(args []string, envs ...string) error {
	cmd := exec.Command(args[0], args[1:]...)
	cmd.Stdout = os.Stdout
	cmd.Stderr = os.Stderr

	cmd.Env = append(os.Environ(), envs...)

	if err := cmd.Run(); err != nil {
		if exiterr, ok := err.(*exec.ExitError); ok {
			if status, ok := exiterr.Sys().(syscall.WaitStatus); ok {
				exitStatus := status.ExitStatus()
				if exitStatus != 1 {
					return xerrors.Errorf("unexpected exit status \"%v\" for %v. failed with %s", exitStatus, cmd.Args, err)
				} else {
					return xerrors.Errorf("unexpected exit status for %v. failed with %s", cmd.Args, err)
				}
			} else {
				return xerrors.Errorf("unexpected exit status for %v. failed with %s", cmd.Args, err)
			}
		} else {
			return xerrors.Errorf("unexpected exit status for %v. failed with %s", cmd.Args, err)
		}
	} else {
		return nil
	}
}

func buildArgs(home, lang, sourcePath, databasePath, command string) []string {
	codeqlCliBinary := path.Join(home, "codeql-cli", "codeql")
	args := []string{codeqlCliBinary, "database", "create", "--language", lang, "--source-root", sourcePath}
	if len(command) != 0 {
		args = append(args, "--command")
		args = append(args, command)
	}
	args = append(args, databasePath)
	return args
}
