package main

import (
	"bytes"
	"fmt"
	"io/ioutil"
	"strings"

	"a.yandex-team.ru/library/go/core/xerrors"
	"a.yandex-team.ru/security/impulse/models"
)

type ResultRegionLocation struct {
	StartLine   int `json:"startLine"`
	EndLine     int `json:"endLine"`
	StartColumn int `json:"startColumn"`
	EndColumn   int `json:"endColumn"`
}

type ResultArtifactLocation struct {
	URI string `json:"uri"`
}

type ResultPhysicalLocation struct {
	ArtifactLocation ResultArtifactLocation `json:"artifactLocation"`
	Region           ResultRegionLocation   `json:"region"`
}

type ResultLocation struct {
	PhysicalLocation ResultPhysicalLocation `json:"physicalLocation"`
}

type ThreadFlowResultLocation struct {
	Location ResultLocation `json:"location"`
}

type ResultMessage struct {
	Text string `json:"text"`
}

type ThreadFlow struct {
	Locations []ThreadFlowResultLocation `json:"locations"`
}

type CodeFlow struct {
	ThreadFlows []ThreadFlow `json:"threadFlows"`
}

type Result struct {
	CodeFlows []CodeFlow       `json:"codeFlows"`
	RuleID    string           `json:"ruleId"`
	Message   ResultMessage    `json:"message"`
	Locations []ResultLocation `json:"locations"`
}

type RuleDescription struct {
	Text string `json:"text"`
}

type RuleProperties struct {
	ID               string   `json:"id"`
	Kind             string   `json:"kind"`
	Precision        string   `json:"precision"`
	SubSeverity      string   `json:"sub-severity"`
	ProblemSeverity  string   `json:"problem.severity"`
	SecuritySeverity string   `json:"security-severity"`
	YandexReferences string   `json:"yandex.refs"`
	Tags             []string `json:"tags"`
}

func (prop RuleProperties) GetReferences() []string {
	results := []string{}
	refs1 := strings.Split(prop.YandexReferences, ",")
	for _, ref1 := range refs1 {
		refs2 := strings.Split(ref1, "\n")
		for _, ref2 := range refs2 {
			ref := strings.TrimSpace(ref2)
			if len(ref) > 0 {
				results = append(results, ref)
			}
		}
	}

	parts := strings.Split(prop.ID, "/")
	lang := ""
	if len(parts) == 2 {
		switch parts[0] {
		case "py":
			lang = "python"
		case "js":
			lang = "javascript"
		case "cs":
			lang = "csharp"
		default:
			lang = parts[0]
		}
		codeqlHelpURL := fmt.Sprintf("https://codeql.github.com/codeql-query-help/%s/%s/", lang, strings.ReplaceAll(prop.ID, "/", "-"))
		results = append(results, codeqlHelpURL)
	}

	return results
}

type Rule struct {
	ID               string          `json:"id"`
	ShortDescription RuleDescription `json:"shortDescription"`
	FullDescription  RuleDescription `json:"fullDescription"`
	Properties       RuleProperties  `json:"properties"`
}

type Driver struct {
	Rules []Rule `json:"rules"`
}

type Tool struct {
	Driver Driver `json:"driver"`
}

type CodeQLRun struct {
	Tool    Tool     `json:"tool"`
	Results []Result `json:"results"`
}

type CodeQLReport struct {
	Runs []CodeQLRun `json:"runs"`
}

func getSeverity(rule Rule) models.SeverityType {
	if rule.Properties.ProblemSeverity == "recommendation" {
		return models.Low

	} else if rule.Properties.ProblemSeverity == "warning" {
		return models.Medium

	} else if rule.Properties.ProblemSeverity == "error" {
		if rule.Properties.SubSeverity == "low" {
			return models.Medium
		} else {
			return models.Critical
		}
	}

	return models.Info
}

func getCode(filePath string, lineNumber int) (*string, error) {
	codeBytes, err := ioutil.ReadFile(filePath)
	if err != nil {
		return nil, err
	}

	codeLines := bytes.Split(codeBytes, []byte("\n"))
	if len(codeLines) < lineNumber {
		return nil, xerrors.Errorf("Source file does not contain %d lines to get code", lineNumber)
	}

	firstLineNumber := 0
	if firstLineNumber < lineNumber-4 {
		firstLineNumber = lineNumber - 4
	}

	lastLineNumber := len(codeLines)
	if lastLineNumber > lineNumber+5 {
		lastLineNumber = lineNumber + 5
	}

	code := bytes.Join(codeLines[firstLineNumber:lastLineNumber], []byte("\n"))
	codeString := string(code)
	return &codeString, nil
}
