package main

import (
	"bytes"
	"encoding/json"
	"log"
	"os"
	"os/exec"
	"path"
	"strings"
	"syscall"

	"a.yandex-team.ru/security/impulse/models"
	"a.yandex-team.ru/security/impulse/workflow/internal/checkout"
	"a.yandex-team.ru/security/impulse/workflow/internal/dedup"
)

type EslintIssueMessage struct {
	RuleID   string `json:"ruleId"`
	Message  string `json:"message"`
	Severity int    `json:"severity"`
	Line     int    `json:"line"`
	EndLine  int    `json:"endLine"`
}

type EslintFileInfo struct {
	FilePath string                `json:"filePath"`
	Source   string                `json:"source"`
	Messages []*EslintIssueMessage `json:"messages"`
}

type Eslint struct {
	basePath       string
	checkoutMeta   *checkout.Checkout
	rawReport      map[string][]byte
	report         []*models.NewVulnerabilityDeduplicationRequestDTO
	extensions     *string
	eslintRcConfig *string
}

var severityMap = map[int]models.SeverityType{
	1: models.Low,
	2: models.Medium,
}

func (r *Eslint) Init(sourcePath string, checkoutMeta *checkout.Checkout) error {
	r.basePath = sourcePath
	r.checkoutMeta = checkoutMeta
	r.rawReport = make(map[string][]byte)

	return nil
}

func (r *Eslint) createEslintConfig() (string, error) {
	file, err := os.OpenFile(DefaultEslintConfigFilename, os.O_WRONLY|os.O_CREATE|os.O_TRUNC, 0666)
	if err != nil {
		return "", err
	}
	defer func() { _ = file.Close() }()
	eslintRcConfig := DefaultEslintConfig
	if *r.eslintRcConfig != "" {
		eslintRcConfig = *r.eslintRcConfig
	}
	_, err = file.WriteString(eslintRcConfig)
	if err != nil {
		return "", err
	}
	return DefaultEslintConfigFilename, nil
}

func (r *Eslint) Run() error {
	eslintConfigFilename, err := r.createEslintConfig()
	if err != nil {
		return err
	}

	for folder := range r.checkoutMeta.Folders {
		sourcePath := path.Join(r.basePath, folder)
		cmd := exec.Command("eslint", "-f", "json", "--no-eslintrc", "--no-inline-config",
			"--ext", *r.extensions, "-c", eslintConfigFilename, sourcePath)
		var stdout bytes.Buffer
		cmd.Stdout = &stdout
		cmd.Stderr = os.Stderr

		r.rawReport[folder] = nil

		if err := cmd.Run(); err != nil {
			if exiterr, ok := err.(*exec.ExitError); ok {
				if status, ok := exiterr.Sys().(syscall.WaitStatus); ok {
					exitStatus := status.ExitStatus()
					if exitStatus == 2 {
						log.Printf("Unexpected exit status \"%v\" for %v\n", exitStatus, cmd.Args)
						r.rawReport[folder], _ = json.Marshal(make([]EslintFileInfo, 0))
						continue
					} else {
						r.rawReport[folder] = stdout.Bytes()
						continue
					}
				}
			} else {
				continue
			}
		}
		r.rawReport[folder] = stdout.Bytes()
	}
	return nil
}

func (r *Eslint) Normalize() error {
	if len(r.rawReport) == 0 {
		r.report = []*models.NewVulnerabilityDeduplicationRequestDTO{}
		return nil
	}

	r.report = make([]*models.NewVulnerabilityDeduplicationRequestDTO, 0)

	for folder, rawReport := range r.rawReport {
		eslintReport := make([]EslintFileInfo, 0)
		err := json.Unmarshal(rawReport, &eslintReport)
		if err != nil {
			log.Println("Could not unmarshal eslint report:", err)
			return err
		}

		for _, eslintFileInfo := range eslintReport {
			filePath := strings.TrimPrefix(eslintFileInfo.FilePath, path.Join(r.basePath, folder))
			sourceStrings := strings.Split(eslintFileInfo.Source, "\n")
			for _, eslintIssueMessage := range eslintFileInfo.Messages {
				if eslintIssueMessage.RuleID == "" {
					continue
				}

				if eslintIssueMessage.Line < 1 {
					eslintIssueMessage.Line = 1
				}
				if eslintIssueMessage.EndLine < eslintIssueMessage.Line {
					eslintIssueMessage.EndLine = eslintIssueMessage.Line
				}

				code := strings.Join(sourceStrings[eslintIssueMessage.Line-1:eslintIssueMessage.EndLine], "\n")
				lineHash, _ := dedup.GetStringHash(code)
				fileURL, _ := checkout.GenerateFileURLWithLineNumber(r.checkoutMeta.Folders[folder], filePath, eslintIssueMessage.Line)
				keyProps := models.VulnerabilityProperties{
					"filename":    filePath,
					"rule":        eslintIssueMessage.RuleID,
					"line_number": eslintIssueMessage.Line,
					"line_hash":   lineHash,
					"description": eslintIssueMessage.Message,
				}
				displayProps := models.VulnerabilityProperties{
					"code":     code,
					"severity": severityMap[eslintIssueMessage.Severity],
					"file_url": fileURL,
				}

				r.report = append(r.report, &models.NewVulnerabilityDeduplicationRequestDTO{
					Severity:          severityMap[eslintIssueMessage.Severity],
					Category:          eslintIssueMessage.RuleID,
					KeyProperties:     keyProps,
					DisplayProperties: displayProps,
				})
			}
		}
	}

	return nil
}

func (r *Eslint) Type() models.ScanTypeName {
	return models.ESLINT
}

func (r *Eslint) RawReport() string {
	rawReport := "[\n"
	for _, rawSingleReport := range r.rawReport {
		rawReport += strings.TrimRight(string(rawSingleReport), "\n") + ",\n"
	}
	if len(rawReport) > 4 {
		rawReport = rawReport[:len(rawReport)-2]
	}
	rawReport += "]"

	return rawReport
}

func (r *Eslint) Report() interface{} {
	return r.report
}

func (r *Eslint) Version() string {
	return "0.1"
}
