package main

import (
	"encoding/xml"
	"io/ioutil"
	"log"
	"os"
	"os/exec"
	"path"
	"path/filepath"
	"strings"
	"syscall"

	"a.yandex-team.ru/library/go/core/xerrors"
	"a.yandex-team.ru/security/impulse/models"
	"a.yandex-team.ru/security/impulse/workflow/internal/checkout"
)

const findSecBugsCmd = "/tmp/findsecbugs.sh"
const findSecBugsReportPath = "/tmp/findsecbugs_report.xml"

type FindSecBugs struct {
	basePath       string
	checkoutMeta   *checkout.Checkout
	rawReport      map[string][]byte
	javaBuildCmd   *string
	javaVersionEnv *string
	report         []*models.NewVulnerabilityDeduplicationRequestDTO
}

type ClassInfo struct {
	Classname  string `json:"classname"`
	SourcePath string `json:"sourcepath"`
	Message    string `json:"message"`
}

type MethodInfo struct {
	Classname  string `json:"classname"`
	MethodName string `json:"method_name"`
	SourcePath string `json:"sourcepath"`
	Message    string `json:"message"`
}

type SourceLineInfo struct {
	Classname  string `json:"classname"`
	Primary    bool   `json:"primary"`
	SourcePath string `json:"sorcepath"`
	StartLine  int    `json:"start_line"`
	EndLine    int    `json:"end_line"`
	Message    string `json:"message"`
	Code       string `json:"code"`
}

type FindSecBugStringInfo struct {
	Message string `json:"message"`
	Role    string `json:"role"`
}

type SourceLine struct {
	XMLName    xml.Name `xml:"SourceLine"`
	Primary    bool     `xml:"primary,attr"`
	Classname  string   `xml:"classname,attr"`
	SourceFile string   `xml:"sourcefile,attr"`
	SourcePath string   `xml:"sourcepath,attr"`
	StartLine  int      `xml:"start,attr"`
	EndLine    int      `xml:"end,attr"`
	Message    string   `xml:"Message"`
}

type FindSecBugString struct {
	XMLName xml.Name `xml:"String"`
	Value   string   `xml:"value,attr"`
	Role    string   `xml:"role,attr"`
	Message string   `xml:"Message"`
}

type Class struct {
	XMLName    xml.Name   `xml:"Class"`
	Classname  string     `xml:"classname,attr"`
	Primary    bool       `xml:"primary,attr"`
	SourceLine SourceLine `xml:"SourceLine"`
	Message    string     `xml:"Message"`
}

type Method struct {
	XMLName    xml.Name   `xml:"Method"`
	Classname  string     `xml:"classname,attr"`
	Name       string     `xml:"name,attr"`
	Primary    bool       `xml:"primary,attr"`
	SourceLine SourceLine `xml:"SourceLine"`
	Message    string     `xml:"Message"`
}

type BugInstance struct {
	XMLName           xml.Name            `xml:"BugInstance"`
	Type              string              `xml:"type,attr"`
	InstanceHash      string              `xml:"instanceHash,attr"`
	Rank              int                 `xml:"rank,attr"`
	Priority          int                 `xml:"priority,attr"`
	CWEID             string              `xml:"cweid,attr"`
	ShortMessage      string              `xml:"ShortMessage"`
	LongMessage       string              `xml:"LongMessage"`
	Classes           []*Class            `xml:"Class"`
	Methods           []*Method           `xml:"Method"`
	SourceLines       []*SourceLine       `xml:"SourceLine"`
	FindSecBugStrings []*FindSecBugString `xml:"String"`
}

func (bug BugInstance) Severity() models.SeverityType {
	switch bug.Rank {
	case 1, 2, 3, 4:
		return models.Critical
	case 5, 6, 7, 8, 9:
		return models.Medium
	case 10, 11, 12, 13, 14:
		return models.Low
	case 15, 16, 17, 18, 19, 20:
		return models.Info
	}
	return models.Info
}

func (bug BugInstance) Confidence() models.ConfidenceType {
	switch bug.Priority {
	case 1:
		return models.ConfidenceHigh
	case 2:
		return models.ConfidenceMedium
	case 3:
		return models.ConfidenceLow
	case 4:
		return models.ConfidenceExperimental
	case 5:
		return models.ConfidenceIgnore
	}
	return models.ConfidenceUnknown
}

type BugPattern struct {
	XMLName          xml.Name `xml:"BugPattern"`
	Type             string   `xml:"type,attr"`
	ShortDescription string   `xml:"ShortDescription"`
	Details          string   `xml:"Details"`
}

type FindSecBugsReport struct {
	XMLName      xml.Name       `xml:"BugCollection"`
	BugInstances []*BugInstance `xml:"BugInstance"`
	BugPatterns  []*BugPattern  `xml:"BugPattern"`
}

func (r *FindSecBugs) Init(sourcePath string, checkoutMeta *checkout.Checkout) error {
	javaHomeValue := os.Getenv(*r.javaVersionEnv)
	if javaHomeValue != "" {
		_ = os.Setenv("JAVA_HOME", javaHomeValue)
	}

	r.basePath = sourcePath
	r.checkoutMeta = checkoutMeta
	r.rawReport = make(map[string][]byte)

	return nil
}

func (r *FindSecBugs) Run() error {
	for folder := range r.checkoutMeta.Folders {
		folderPath := path.Join(r.basePath, folder)
		cmdLineArray := strings.Split(strings.TrimSpace(*r.javaBuildCmd), " ")

		if len(cmdLineArray) == 0 || cmdLineArray[0] == "" {
			log.Printf("No valid cmdline for building jar from sources %s\n", *r.javaBuildCmd)
			return nil
		}
		cmd := exec.Command(cmdLineArray[0], cmdLineArray[1:]...)
		cmd.Dir = folderPath
		cmd.Stdout = os.Stdout
		cmd.Stderr = os.Stderr
		if err := cmd.Run(); err != nil {
			if exiterr, ok := err.(*exec.ExitError); ok {
				if status, ok := exiterr.Sys().(syscall.WaitStatus); ok {
					exitStatus := status.ExitStatus()
					if exitStatus != 0 {
						log.Printf("Unexpected exit status of building \"%v\" for %v\n", exitStatus, cmd.Args)
					}
				}
			}

		}

		allJarFiles := make([]string, 0)
		err := filepath.Walk(folderPath, func(path string, info os.FileInfo, err error) error {
			if err != nil {
				return err
			}
			if info.IsDir() {
				return nil
			}
			if strings.HasSuffix(path, ".jar") {
				if strings.HasSuffix(path, "tests.jar") {
					// ignore tests
					return nil
				}
				allJarFiles = append(allJarFiles, path)
				return nil
			}

			return nil
		})
		if err != nil {
			return err
		}

		findSecBugsArguments := []string{"-sortByClass", "-dontCombineWarnings",
			"-progress", "-xml:withMessages", "-output", findSecBugsReportPath}
		findSecBugsArguments = append(findSecBugsArguments, allJarFiles...)
		findSecBugsCmd := exec.Command(findSecBugsCmd, findSecBugsArguments...)
		findSecBugsCmd.Dir = folderPath
		findSecBugsCmd.Stdout = os.Stdout
		findSecBugsCmd.Stderr = os.Stderr
		if err := findSecBugsCmd.Run(); err != nil {
			if exiterr, ok := err.(*exec.ExitError); ok {
				if status, ok := exiterr.Sys().(syscall.WaitStatus); ok {
					exitStatus := status.ExitStatus()
					if exitStatus != 0 {
						log.Printf("Unexpected exit status of findsecbugs \"%v\" for %v\n", exitStatus,
							findSecBugsCmd.Args)
					}
				}
			}

		}

		r.rawReport[folder], _ = ioutil.ReadFile(findSecBugsReportPath)
	}

	return nil
}

func (r *FindSecBugs) findFullFilePath(folder string, endFileName string) (string, error) {
	prefix := path.Join(r.basePath, folder)
	fullFilePath := ""
	err := filepath.Walk(prefix, func(path string, info os.FileInfo, err error) error {
		if err != nil {
			return err
		}
		if info.IsDir() {
			return nil
		}
		if strings.HasSuffix(path, endFileName) {
			fullFilePath = path
			return nil
		}

		return nil
	})
	if err != nil {
		return "", nil
	}
	return fullFilePath, nil
}

func (r *FindSecBugs) readCodeFromFile(filename string, startLine int, endLine int) (string, error) {
	file, err := ioutil.ReadFile(filename)
	if err != nil {
		return "", err
	}
	allCode := strings.Split(string(file), "\n")
	if startLine < 1 || startLine > len(allCode) || endLine < 1 || endLine > len(allCode) {
		return "", xerrors.Errorf("invalid startLine (%d) or endLine (%d)", startLine, endLine)
	}
	return strings.Join(allCode[startLine-1:endLine], "\n"), nil
}

func (r *FindSecBugs) Normalize() error {
	if len(r.rawReport) == 0 {
		return xerrors.Errorf("FindSecBugs has generated empty rawReport")
	}

	r.report = make([]*models.NewVulnerabilityDeduplicationRequestDTO, 0)

	for folder, rawReport := range r.rawReport {
		var findSecBugsReport FindSecBugsReport
		if err := xml.Unmarshal(rawReport, &findSecBugsReport); err != nil {
			return err
		}

		bugPatternsMap := make(map[string]*BugPattern)
		for _, bugPattern := range findSecBugsReport.BugPatterns {
			bugPatternsMap[bugPattern.Type] = bugPattern
		}
		for _, bugInstance := range findSecBugsReport.BugInstances {

			var primaryClass ClassInfo
			secondaryClasses := make([]ClassInfo, 0)
			for _, class := range bugInstance.Classes {

				fullSourcePath, err := r.findFullFilePath(folder, class.SourceLine.SourcePath)
				if err != nil {
					return err
				}
				filePath := strings.TrimPrefix(fullSourcePath, path.Join(r.basePath, folder))
				if class.Primary {
					primaryClass = ClassInfo{
						Classname:  class.Classname,
						SourcePath: filePath,
						Message:    class.Message,
					}
				} else {
					secondaryClasses = append(secondaryClasses, ClassInfo{
						Classname:  class.Classname,
						SourcePath: filePath,
						Message:    class.Message,
					})
				}
			}
			if primaryClass.Classname == "" && len(secondaryClasses) > 0 {
				primaryClass = secondaryClasses[0]
			}

			var primaryMethod MethodInfo
			secondaryMethods := make([]MethodInfo, 0)
			for _, method := range bugInstance.Methods {
				fullSourcePath, err := r.findFullFilePath(folder, method.SourceLine.SourcePath)
				if err != nil {
					return err
				}
				filePath := strings.TrimPrefix(fullSourcePath, path.Join(r.basePath, folder))
				if method.Primary {
					primaryMethod = MethodInfo{
						Classname:  method.Classname,
						MethodName: method.Name,
						SourcePath: filePath,
						Message:    method.Message,
					}
				} else {
					secondaryMethods = append(secondaryMethods, MethodInfo{
						Classname:  method.Classname,
						MethodName: method.Name,
						SourcePath: filePath,
						Message:    method.Message,
					})
				}
			}
			if primaryMethod.MethodName == "" && len(secondaryMethods) > 0 {
				primaryMethod = secondaryMethods[0]
			}

			primaryLineNumber := -1
			sourceLines := make([]SourceLineInfo, 0)
			for _, line := range bugInstance.SourceLines {
				fullSourcePath, err := r.findFullFilePath(folder, line.SourcePath)
				if err != nil {
					fullSourcePath = ""
				}
				filePath := strings.TrimPrefix(fullSourcePath, path.Join(r.basePath, folder))
				code, err := r.readCodeFromFile(fullSourcePath, line.StartLine, line.EndLine)
				if err != nil {
					code = ""
				}
				sourceLines = append(sourceLines, SourceLineInfo{
					Classname:  line.Classname,
					Primary:    line.Primary,
					SourcePath: filePath,
					StartLine:  line.StartLine,
					EndLine:    line.EndLine,
					Message:    line.Message,
					Code:       code,
				})
				if line.Primary {
					primaryLineNumber = line.StartLine
				}
			}
			if primaryLineNumber == -1 {
				if len(sourceLines) > 0 {
					primaryLineNumber = sourceLines[0].StartLine
				} else {
					continue
				}
			}

			findSecBugsStrings := make([]FindSecBugStringInfo, 0)
			for _, findSecBugsString := range bugInstance.FindSecBugStrings {
				findSecBugsStrings = append(findSecBugsStrings, FindSecBugStringInfo{
					Message: findSecBugsString.Message,
					Role:    findSecBugsString.Role,
				})
			}

			if primaryClass.SourcePath == "" {
				// ignore vulns in 3d-patry libraries
				continue
			}

			fileURL, _ := checkout.GenerateFileURLWithLineNumber(r.checkoutMeta.Folders[folder],
				primaryClass.SourcePath, primaryLineNumber)
			bugPattern, bugPatterExists := bugPatternsMap[bugInstance.Type]
			bugPatternShortDescription, bugPatternDetails := "", ""
			if bugPatterExists {
				bugPatternShortDescription = bugPattern.ShortDescription
				bugPatternDetails = bugPattern.Details
			}

			keyProps := models.VulnerabilityProperties{
				"instance_hash":     bugInstance.InstanceHash,
				"type":              bugInstance.Type,
				"primary_classname": primaryClass.Classname,
				"filename":          primaryClass.SourcePath,
			}

			displayProps := models.VulnerabilityProperties{
				"confidence":            bugInstance.Confidence(),
				"cweid":                 bugInstance.CWEID,
				"short_message":         bugInstance.ShortMessage,
				"long_message":          bugInstance.LongMessage,
				"short_description":     bugPatternShortDescription,
				"details":               bugPatternDetails,
				"primary_class":         primaryClass,
				"primary_method":        primaryMethod,
				"secondary_classes":     secondaryClasses,
				"secondary_methods":     secondaryMethods,
				"find_sec_bugs_strings": findSecBugsStrings,
				"file_url":              fileURL,
				"sourcelines":           sourceLines,
			}

			r.report = append(r.report, &models.NewVulnerabilityDeduplicationRequestDTO{
				Severity:          bugInstance.Severity(),
				Category:          bugInstance.Type,
				KeyProperties:     keyProps,
				DisplayProperties: displayProps,
			})
		}
	}

	return nil
}

func (r *FindSecBugs) Type() models.ScanTypeName {
	return models.FINDSECBUGS
}

func (r *FindSecBugs) Report() interface{} {
	return r.report
}

func (r *FindSecBugs) RawReport() string {
	rawReport := "[\n"
	for _, rawSingleReport := range r.rawReport {
		rawReport += strings.TrimRight(string(rawSingleReport), "\n") + ",\n"
	}
	if len(rawReport) > 4 {
		rawReport = rawReport[:len(rawReport)-2]
	}
	rawReport += "]"

	return rawReport
}

func (r *FindSecBugs) Version() string {
	return "0.1"
}
