package main

import (
	"bytes"
	"encoding/json"
	"log"
	"os"
	"os/exec"
	"path"
	"strconv"
	"strings"
	"syscall"

	"a.yandex-team.ru/security/impulse/models"
	"a.yandex-team.ru/security/impulse/workflow/internal/checkout"
	"a.yandex-team.ru/security/impulse/workflow/internal/dedup"
)

type GosecVuln struct {
	Severity   string `json:"severity"`
	Confidence string `json:"confidence"`
	RuleID     string `json:"rule_id"`
	Details    string `json:"details"`
	File       string `json:"file"`
	Code       string `json:"code"`
	Line       string `json:"line"`
}

type GosecReport struct {
	Issues []GosecVuln `json:"issues"`
}

type Gosec struct {
	basePath     string
	checkoutMeta *checkout.Checkout
	rawReport    map[string][]byte
	report       []*models.NewVulnerabilityDeduplicationRequestDTO
}

var severityMap = map[string]models.SeverityType{
	"LOW":    models.Low,
	"MEDIUM": models.Medium,
	"HIGH":   models.Critical,
}

func (r *Gosec) Init(sourcePath string, checkoutMeta *checkout.Checkout) error {
	_ = os.Setenv("GOPATH", sourcePath)
	r.basePath = sourcePath
	r.checkoutMeta = checkoutMeta
	r.rawReport = make(map[string][]byte)

	return nil
}

func (r *Gosec) Run() error {
	for folder := range r.checkoutMeta.Folders {
		sourcePath := path.Join(r.basePath, folder)
		cmd := exec.Command("gosec", "-fmt", "json", sourcePath+"/...")
		var stdout bytes.Buffer
		cmd.Stdout = &stdout
		cmd.Stderr = os.Stderr

		r.rawReport[folder] = nil

		if err := cmd.Run(); err != nil {
			if exiterr, ok := err.(*exec.ExitError); ok {
				if status, ok := exiterr.Sys().(syscall.WaitStatus); ok {
					exitStatus := status.ExitStatus()
					if exitStatus != 1 {
						log.Printf("Unexpected exit status \"%v\" for %v\n", exitStatus, cmd.Args)
					} else {
						r.rawReport[folder] = stdout.Bytes()
					}
				}
			}
		} else {
			r.rawReport[folder] = stdout.Bytes()
		}
	}
	return nil
}

func (r *Gosec) Normalize() error {
	if len(r.rawReport) == 0 {
		r.report = []*models.NewVulnerabilityDeduplicationRequestDTO{}
		return nil
	}

	r.report = make([]*models.NewVulnerabilityDeduplicationRequestDTO, 0)

	for folder, rawReport := range r.rawReport {
		report := GosecReport{}
		err := json.Unmarshal(rawReport, &report)
		if err != nil {
			log.Println("Could not unmarshal gosec report:", err)
			continue
		}

		for _, gosecVuln := range report.Issues {
			filePath := strings.TrimPrefix(gosecVuln.File, path.Join(r.basePath, folder))
			lineNumber, err := strconv.Atoi(strings.Split(gosecVuln.Line, "-")[0])
			if err != nil {
				log.Printf("Could not convert %v to number: %v\n", gosecVuln.Line, err)
				continue
			}
			hash, _ := dedup.GetLineHash(gosecVuln.File, lineNumber)
			fileURL, _ := checkout.GenerateFileURLWithLineNumber(r.checkoutMeta.Folders[folder], filePath, lineNumber)

			keyProps := models.VulnerabilityProperties{
				"filename":    filePath,
				"rule":        gosecVuln.RuleID,
				"line_number": lineNumber,
				"line_hash":   hash,
				"description": gosecVuln.Details,
			}
			displayProps := models.VulnerabilityProperties{
				"code":     gosecVuln.Code,
				"severity": severityMap[gosecVuln.Severity],
				"file_url": fileURL,
			}

			r.report = append(r.report, &models.NewVulnerabilityDeduplicationRequestDTO{
				Severity:          severityMap[gosecVuln.Severity],
				Category:          gosecVuln.Details,
				KeyProperties:     keyProps,
				DisplayProperties: displayProps,
			})
		}
	}

	return nil
}

func (r *Gosec) Type() models.ScanTypeName {
	return models.GOSEC
}

func (r *Gosec) RawReport() string {
	rawReport := "[\n"
	for _, rawSingleReport := range r.rawReport {
		rawReport += strings.TrimRight(string(rawSingleReport), "\n") + ",\n"
	}
	if len(rawReport) > 4 {
		rawReport = rawReport[:len(rawReport)-2]
	}
	rawReport += "]"

	return rawReport
}

func (r *Gosec) Report() interface{} {
	return r.report
}

func (r *Gosec) Version() string {
	return "0.1"
}
