package main

import (
	"bytes"
	"encoding/json"
	"fmt"
	"io"
	"io/ioutil"
	"log"
	"mime/multipart"
	"net/http"
	"os"
	"path"
	"path/filepath"
	"time"

	"golang.org/x/xerrors"

	"a.yandex-team.ru/security/impulse/models"
	"a.yandex-team.ru/security/impulse/workflow/internal/checkout"
	"a.yandex-team.ru/security/impulse/workflow/internal/compress"
)

type Rips struct {
	basePath      string
	checkoutMeta  *checkout.Checkout
	upstreamURL   *string
	upstreamUIURL *string
	username      *string
	password      *string
	appID         *int
	scanID        int
	issues        []RipsIssue
	issueTypes    map[int]RipsIssueType
	report        []*models.NewVulnerabilityDeduplicationRequestDTO
	rawReport     []byte
}

type RipsUpload struct {
	ID int `json:"id"`
}

type RipsScanStatus struct {
	ID      int `json:"id"`
	Percent int `json:"percent"`
}

type RipsIssueType struct {
	ID          int    `json:"id"`
	Description string `json:"description"`
	Name        string `json:"name"`
	Severity    int    `json:"severity"`
}

type RipsIssueNestedType struct {
	ID int `json:"id"`
}

type RipsFile struct {
	ID   int    `json:"id"`
	Path string `json:"path"`
}

type RipsSink struct {
	ID        int      `json:"id"`
	StartLine int      `json:"start_line"`
	File      RipsFile `json:"file"`
}

type RipsIssue struct {
	ID   int                 `json:"id"`
	Type RipsIssueNestedType `json:"type"`
	Hash string              `json:"hash"`
	UUID string              `json:"uuid"`
	Sink RipsSink            `json:"sink"`
}

func (r *Rips) Init(sourcePath string, checkoutMeta *checkout.Checkout) error {
	r.basePath = sourcePath
	r.checkoutMeta = checkoutMeta

	return nil
}

func (r *Rips) Run() error {
	log.Println(r.basePath)
	// compress Zip
	zipFile, err := ioutil.TempFile("/tmp", "rips_*.zip")
	if err != nil {
		return err
	}

	defer func() { _ = os.Remove(zipFile.Name()) }()
	var paths []string
	for folder := range r.checkoutMeta.Folders {
		paths = append(paths, path.Join(r.basePath, folder))
	}
	log.Println("Start compress source code to zipFile:", zipFile.Name())
	err = compress.ZipFilesToFile(paths, r.basePath, zipFile, []string{".git", ".svn"})
	if err != nil {
		return err
	}
	log.Println("Compress is successful:", zipFile.Name())
	// Upload Zip to RIPS
	ripsUpload := RipsUpload{}
	err = r.uploadZip(zipFile, &ripsUpload)
	if err != nil {
		return err
	}
	if ripsUpload.ID == 0 {
		return xerrors.Errorf("RIPS upload error")
	}
	log.Println("Uploaded. UploadId:", ripsUpload.ID)

	scanStatus := RipsScanStatus{}
	err = r.startScan(time.Now().Format("2006-01-02"), ripsUpload.ID, &scanStatus)
	if err != nil {
		return err
	}
	if scanStatus.ID == 0 {
		return xerrors.Errorf("RIPS scan start error")
	}
	log.Println("Start Scan. ScanId:", scanStatus.ID)

	// wait until scan not finished
	for oldScanStatusPercent := -1; scanStatus.Percent != 100; {
		time.Sleep(10 * time.Second)
		err = r.scanStatus(scanStatus.ID, &scanStatus)
		if err != nil {
			return err
		}
		if oldScanStatusPercent < scanStatus.Percent {
			log.Println("Scan progress (%):", scanStatus.Percent)
			oldScanStatusPercent = scanStatus.Percent
		}
	}

	// get Issues Types
	issueTypes, err := r.getIssueTypes()
	if err != nil {
		return nil
	}
	r.issueTypes = make(map[int]RipsIssueType)
	for _, issue := range issueTypes {
		r.issueTypes[issue.ID] = issue
	}
	// get Scan stat
	r.scanID = scanStatus.ID
	issuesCount, err := r.getScanIssuesCount(scanStatus.ID)
	if err != nil {
		return err
	}
	// get Issues
	for i := 0; i < issuesCount; i += 500 {
		var newIssues []RipsIssue
		newIssues, err = r.getIssues(scanStatus.ID, 500, i)
		if err != nil {
			return err
		}
		r.issues = append(r.issues, newIssues...)
	}

	r.rawReport, err = json.Marshal(r.issues)
	if err != nil {
		return err
	}
	return nil
}

func (r *Rips) Normalize() error {
	for _, ripsIssue := range r.issues {
		fileURL, _ := checkout.GenerateFileURLWithLineNumberByFolders(r.checkoutMeta.Folders, ripsIssue.Sink.File.Path,
			ripsIssue.Sink.StartLine)

		keyProps := models.VulnerabilityProperties{
			"uuid":     ripsIssue.UUID,
			"filename": ripsIssue.Sink.File.Path,
		}

		referenceRipsURL := fmt.Sprintf("%s/issue/%d/%d/%d/%d/details", *r.upstreamUIURL, *r.appID, r.scanID,
			ripsIssue.Type.ID, ripsIssue.ID)

		displayProps := models.VulnerabilityProperties{
			"filename":    ripsIssue.Sink.File.Path,
			"description": r.issueTypes[ripsIssue.Type.ID].Description,
			"file_url":    fileURL,
			"hash":        ripsIssue.Hash,
			"reference":   referenceRipsURL,
		}

		var issueSeverity models.SeverityType
		switch {
		case r.issueTypes[ripsIssue.Type.ID].Severity >= 75:
			issueSeverity = models.Critical
		case r.issueTypes[ripsIssue.Type.ID].Severity >= 50:
			issueSeverity = models.Medium
		case r.issueTypes[ripsIssue.Type.ID].Severity >= 25:
			issueSeverity = models.Low
		default:
			issueSeverity = models.Info
		}

		r.report = append(r.report, &models.NewVulnerabilityDeduplicationRequestDTO{
			Severity:          issueSeverity,
			Category:          r.issueTypes[ripsIssue.Type.ID].Name,
			KeyProperties:     keyProps,
			DisplayProperties: displayProps,
		})
	}

	return nil
}

func (r *Rips) Type() models.ScanTypeName {
	return models.RIPS
}

func (r *Rips) RawReport() string {
	return string(r.rawReport)
}

func (r *Rips) Report() interface{} {
	return r.report
}

func (r *Rips) setAuthHeaders(req *http.Request) {
	req.Header.Set("X-API-Username", *r.username)
	req.Header.Set("X-API-Password", *r.password)
}

func (r *Rips) uploadZip(zipFilename *os.File, targetUpload interface{}) error {
	uri := fmt.Sprintf("%s/applications/%d/uploads", *r.upstreamURL, *r.appID)
	req, err := newfileUploadRequest(uri, nil, "upload[file]", zipFilename.Name())
	if err != nil {
		return err
	}

	r.setAuthHeaders(req)

	client := &http.Client{}
	resp, err := client.Do(req)
	if err != nil {
		return err
	}
	defer func() { _ = resp.Body.Close() }()

	body, err := ioutil.ReadAll(resp.Body)
	if err != nil {
		return err
	}
	log.Println(string(body))

	err = json.Unmarshal(body, &targetUpload)
	if err != nil {
		return err
	}

	return nil
}

func (r *Rips) startScan(scanVersion string, uploadID int, targetStatus interface{}) error {
	type ScanVersion struct {
		Version  string `json:"version"`
		UploadID int    `json:"upload"`
	}
	type ScanUpload struct {
		Scan ScanVersion `json:"scan"`
	}
	body := &ScanUpload{Scan: ScanVersion{Version: scanVersion, UploadID: uploadID}}
	buf := new(bytes.Buffer)
	err := json.NewEncoder(buf).Encode(body)
	if err != nil {
		return err
	}

	uri := fmt.Sprintf("%s/applications/%d/scans", *r.upstreamURL, *r.appID)
	req, err := http.NewRequest("POST", uri, buf)
	if err != nil {
		return err
	}

	req.Header.Set("Content-Type", "application/json")
	r.setAuthHeaders(req)

	client := &http.Client{}
	resp, err := client.Do(req)
	if err != nil {
		return err
	}
	defer func() { _ = resp.Body.Close() }()

	respBody, err := ioutil.ReadAll(resp.Body)
	if err != nil {
		return err
	}

	err = json.Unmarshal(respBody, &targetStatus)
	if err != nil {
		return err
	}
	return nil
}

func (r *Rips) scanStatus(scanID int, targetStatus interface{}) error {
	uri := fmt.Sprintf("%s/applications/%d/scans/%d", *r.upstreamURL, *r.appID, scanID)
	req, err := http.NewRequest("GET", uri, nil)
	if err != nil {
		return err
	}

	r.setAuthHeaders(req)

	client := &http.Client{}
	resp, err := client.Do(req)
	if err != nil {
		return err
	}
	defer func() { _ = resp.Body.Close() }()

	body, err := ioutil.ReadAll(resp.Body)
	if err != nil {
		return err
	}

	err = json.Unmarshal(body, &targetStatus)
	if err != nil {
		return err
	}
	return nil
}

func (r *Rips) getScanIssuesCount(scanID int) (int, error) {
	type ScanIssuesResp struct {
		IssuesCount int `json:"issues"`
	}
	uri := fmt.Sprintf(
		`%s/applications/%d/scans/%d/issues/stats?filter={"__equal":{"negativelyReviewed":"0"}}`,
		*r.upstreamURL, *r.appID, scanID,
	)
	req, err := http.NewRequest("GET", uri, nil)
	if err != nil {
		return 0, err
	}
	r.setAuthHeaders(req)

	client := &http.Client{}
	resp, err := client.Do(req)
	if err != nil {
		return 0, err
	}
	defer func() { _ = resp.Body.Close() }()

	body, err := ioutil.ReadAll(resp.Body)
	if err != nil {
		return 0, err
	}

	var scanIssuesResp ScanIssuesResp
	err = json.Unmarshal(body, &scanIssuesResp)
	if err != nil {
		return 0, err
	}

	return scanIssuesResp.IssuesCount, nil
}

func (r *Rips) getIssues(scanID int, limit int, offset int) ([]RipsIssue, error) {
	uri := fmt.Sprintf(
		`%s/applications/%d/scans/%d/issues?limit=%d&offset=%d`+
			`&filter={"__or":[{"__equal":{"negativelyReviewed":"0"}}]}&select=["id","hash","uuid","sink","type"]`,
		*r.upstreamURL, *r.appID, scanID, limit, offset,
	)
	req, err := http.NewRequest("GET", uri, nil)
	if err != nil {
		return nil, err
	}
	r.setAuthHeaders(req)

	client := &http.Client{}
	resp, err := client.Do(req)
	if err != nil {
		return nil, err
	}
	defer func() { _ = resp.Body.Close() }()

	body, err := ioutil.ReadAll(resp.Body)
	if err != nil {
		return nil, err
	}

	var issues []RipsIssue
	err = json.Unmarshal(body, &issues)
	if err != nil {
		return nil, err
	}

	return issues, nil
}

func (r *Rips) getIssueTypes() ([]RipsIssueType, error) {
	uri := fmt.Sprintf(
		`%s/applications/scans/issues/types?select=["id","description","name","severity"]`,
		*r.upstreamURL,
	)
	req, err := http.NewRequest("GET", uri, nil)
	if err != nil {
		return nil, err
	}
	r.setAuthHeaders(req)

	client := &http.Client{}
	resp, err := client.Do(req)
	if err != nil {
		return nil, err
	}
	defer func() { _ = resp.Body.Close() }()

	body, err := ioutil.ReadAll(resp.Body)
	if err != nil {
		return nil, err
	}

	var issuesTypes []RipsIssueType
	err = json.Unmarshal(body, &issuesTypes)
	if err != nil {
		return nil, err
	}

	return issuesTypes, nil
}

func newfileUploadRequest(uri string, params map[string]string, paramName, filePath string) (*http.Request, error) {
	file, err := os.Open(filePath)
	if err != nil {
		return nil, err
	}
	defer func() { _ = file.Close() }()

	body := &bytes.Buffer{}
	writer := multipart.NewWriter(body)
	part, err := writer.CreateFormFile(paramName, filepath.Base(filePath))
	if err != nil {
		return nil, err
	}
	_, err = io.Copy(part, file)
	if err != nil {
		return nil, err
	}

	for key, val := range params {
		_ = writer.WriteField(key, val)
	}
	err = writer.Close()
	if err != nil {
		return nil, err
	}

	req, err := http.NewRequest("POST", uri, body)
	if err != nil {
		return nil, err
	}
	req.Header.Set("Content-Type", writer.FormDataContentType())
	return req, err
}

func (r *Rips) Version() string {
	return "0.1"
}
