package main

import (
	"bytes"
	"encoding/json"
	"log"
	"os"
	"os/exec"
	"path"
	"strconv"
	"strings"
	"syscall"
	"time"

	"a.yandex-team.ru/security/impulse/models"
	"a.yandex-team.ru/security/impulse/workflow/internal/checkout"
	"a.yandex-team.ru/security/impulse/workflow/internal/dedup"
)

const semgrepRulesRepo = "https://bb.yandex-team.ru/scm/sec/semgrep-rules.git"
const semgrepRulesBranch = "yamaster"
const semgrepRulesFilePath = "/tmp/semgrep_testing"
const semgrepRulesPrefix = "tmp.semgrep_testing"
const semgrepRulesPostfix = ".semgrep-rules.git."
const semgrepMaxCodeLen = 10000

type SemgrepIssue struct {
	CheckID string `json:"check_id"`
	Path    string `json:"path"`
	Start   struct {
		Line int `json:"line"`
		Col  int `json:"col"`
	} `json:"start"`
	End struct {
		Line int `json:"line"`
		Col  int `json:"col"`
	} `json:"end"`
	Extra struct {
		Message  string `json:"message"`
		Severity string `json:"severity"`
		Lines    string `json:"lines"`
		Metadata struct {
			RuleTitle     string   `json:"rule-title"`
			Severity      string   `json:"severity"`
			Confidence    string   `json:"confidence"`
			Cwe           string   `json:"cwe"`
			CvssScore     string   `json:"cvss-score"`
			CvssVector    string   `json:"cvss-vector"`
			SourceRuleURL string   `json:"source-rule-url"`
			References    []string `json:"references"`
		} `json:"metadata"`
	} `json:"extra"`
}

func (bug SemgrepIssue) Severity() models.SeverityType {
	if bug.Extra.Metadata.Severity != "" {
		var severityMap = map[string]models.SeverityType{
			"INFO":     models.Info,
			"LOW":      models.Low,
			"MEDIUM":   models.Medium,
			"CRITICAL": models.Critical,
			"BLOCKER":  models.Blocker,
		}
		return severityMap[bug.Extra.Metadata.Severity]
	} else {
		var severityMap = map[string]models.SeverityType{
			"INFO":    models.Info,
			"WARNING": models.Low,
			"ERROR":   models.Medium,
		}
		return severityMap[bug.Extra.Severity]
	}
}

func (bug SemgrepIssue) RuleID() string {
	return bug.CheckID
}

func (bug SemgrepIssue) Category() string {
	if bug.Extra.Metadata.RuleTitle != "" {
		return bug.Extra.Metadata.RuleTitle
	} else {
		return bug.RuleID()
	}
}

type SemgrepReport struct {
	Results []SemgrepIssue `json:"results"`
}

type Semgrep struct {
	basePath       string
	checkoutMeta   *checkout.Checkout
	rawReport      map[string]map[string][]byte
	report         []*models.NewVulnerabilityDeduplicationRequestDTO
	rawConfigPaths *string
	configPaths    []string
	rulesFolder    string
	rulesPrefix    string
}

func (r *Semgrep) Init(sourcePath string, checkoutMeta *checkout.Checkout) error {
	r.basePath = sourcePath
	r.checkoutMeta = checkoutMeta
	r.rawReport = make(map[string]map[string][]byte)

	err := json.Unmarshal([]byte(*r.rawConfigPaths), &r.configPaths)
	if err != nil {
		log.Printf("Could not unmarshal include patterns %s: %v\n", *r.rawConfigPaths, err)
		return err
	}

	currentTimeStrPart := strconv.FormatInt(time.Now().Unix(), 10)
	currentSemgrepRulesFilePath := semgrepRulesFilePath + currentTimeStrPart
	rulesFolder, _, err := checkout.GitClone(semgrepRulesRepo, semgrepRulesBranch,
		currentSemgrepRulesFilePath)
	r.rulesFolder = path.Join(currentSemgrepRulesFilePath, rulesFolder)
	r.rulesPrefix = semgrepRulesPrefix + currentTimeStrPart + semgrepRulesPostfix
	if err != nil {
		return err
	}
	return nil
}

func (r *Semgrep) Run() error {
	for folder := range r.checkoutMeta.Folders {
		sourcePath := path.Join(r.basePath, folder)
		r.rawReport[folder] = make(map[string][]byte)

		for _, configPath := range r.configPaths {
			fullConfigPath := path.Join(r.rulesFolder, configPath)
			_ = os.Setenv("LANG", "C.UTF-8")
			cmd := exec.Command("semgrep", "--dangerously-allow-arbitrary-code-execution-from-rules",
				"--quiet", "--disable-version-check", "--timeout", "10", "--timeout-threshold", "2",
				"--jobs", "1", "--config", fullConfigPath, "--json", sourcePath)
			var stdout bytes.Buffer
			cmd.Stdout = &stdout
			cmd.Stderr = os.Stderr

			r.rawReport[folder][configPath] = nil

			if err := cmd.Run(); err != nil {
				if exiterr, ok := err.(*exec.ExitError); ok {
					if status, ok := exiterr.Sys().(syscall.WaitStatus); ok {
						exitStatus := status.ExitStatus()
						log.Printf("Unexpected exit status \"%v\" for %v\n", exitStatus, cmd.Args)
						r.rawReport[folder][configPath] = stdout.Bytes()
						continue
					}
				} else {
					r.rawReport[folder][configPath], _ = json.Marshal(SemgrepReport{})
					continue
				}
			}
			r.rawReport[folder][configPath] = stdout.Bytes()
		}
	}

	return nil
}

func (r *Semgrep) Normalize() error {
	if len(r.rawReport) == 0 {
		r.report = []*models.NewVulnerabilityDeduplicationRequestDTO{}
		return nil
	}

	r.report = make([]*models.NewVulnerabilityDeduplicationRequestDTO, 0)

	for folder, configRawReport := range r.rawReport {
		for _, RawReport := range configRawReport {
			var semgrepReport SemgrepReport
			err := json.Unmarshal(RawReport, &semgrepReport)
			if err != nil {
				log.Println("Could not unmarshal semgrep report:", err)
				return err
			}

			for _, semgrepIssue := range semgrepReport.Results {
				filePath := strings.TrimPrefix(semgrepIssue.Path, path.Join(r.basePath, folder))
				fileURL, _ := checkout.GenerateFileURLWithLineNumber(r.checkoutMeta.Folders[folder], filePath,
					semgrepIssue.Start.Line)
				codeLines := semgrepIssue.Extra.Lines
				if len(codeLines) > semgrepMaxCodeLen {
					codeLines = codeLines[:semgrepMaxCodeLen]
				}
				lineHash, _ := dedup.GetStringHash(codeLines)
				semgrepIssue.CheckID = strings.TrimPrefix(semgrepIssue.CheckID, r.rulesPrefix)

				keyProps := models.VulnerabilityProperties{
					"filename":    filePath,
					"rule":        semgrepIssue.RuleID(),
					"line_number": semgrepIssue.Start.Line,
					"line_hash":   lineHash,
					"description": semgrepIssue.Extra.Message,
				}
				displayProps := models.VulnerabilityProperties{
					"code":            codeLines,
					"severity":        semgrepIssue.Severity(),
					"file_url":        fileURL,
					"confidence":      semgrepIssue.Extra.Metadata.Confidence,
					"cwe":             semgrepIssue.Extra.Metadata.Cwe,
					"cvss_score":      semgrepIssue.Extra.Metadata.CvssScore,
					"cvss_vector":     semgrepIssue.Extra.Metadata.CvssVector,
					"source_rule_url": semgrepIssue.Extra.Metadata.SourceRuleURL,
					"references":      semgrepIssue.Extra.Metadata.References,
				}
				r.report = append(r.report, &models.NewVulnerabilityDeduplicationRequestDTO{
					Severity:          semgrepIssue.Severity(),
					Category:          semgrepIssue.Category(),
					KeyProperties:     keyProps,
					DisplayProperties: displayProps,
				})
			}
		}
	}
	return nil
}

func (r *Semgrep) Type() models.ScanTypeName {
	return models.SEMGREP
}

func (r *Semgrep) RawReport() string {
	rawReport := "[\n"
	for _, rawFolderReports := range r.rawReport {
		for _, rawSingleReport := range rawFolderReports {
			rawReport += strings.TrimRight(string(rawSingleReport), "\n") + ",\n"
		}
	}
	if len(rawReport) > 4 {
		rawReport = rawReport[:len(rawReport)-2]
	}
	rawReport += "]"
	return rawReport
}

func (r *Semgrep) Report() interface{} {
	return r.report
}

func (r *Semgrep) Version() string {
	return "0.1"
}
