package main

import (
	"bytes"
	"encoding/json"
	"fmt"
	"io/ioutil"
	"log"
	"os"
	"os/exec"
	"path"
	"regexp"
	"strings"
	"syscall"

	"a.yandex-team.ru/security/impulse/models"
	"a.yandex-team.ru/security/impulse/workflow/internal/checkout"
)

type YadiIssue struct {
	ID                 string   `json:"id"`
	PackageName        string   `json:"package_name"`
	VulnerableVersions string   `json:"vulnerable_versions"`
	PatchedVersions    string   `json:"patched_versions"`
	Summary            string   `json:"summary"`
	Reference          string   `json:"reference"`
	CvssScore          float64  `json:"cvss_score"`
	PatchExists        bool     `json:"patch_exists"`
	Version            string   `json:"version"`
	Suggest            []string `json:"suggest"`
	Path               []string `json:"path"`
}

type YadiReport struct {
	Path   string       `json:"path"`
	Owners []string     `json:"owners"`
	Issues []*YadiIssue `json:"issues"`
}

type Yadi struct {
	basePath     string
	feed         *string
	runVeendor   *int
	checkoutMeta *checkout.Checkout
	rawReport    map[string][]byte
	report       []*models.NewVulnerabilityDeduplicationRequestDTO
}

func cvssToSeverity(cvss float64) models.SeverityType {
	switch {
	case cvss < 4.0:
		return models.Low
	case cvss < 7.0:
		return models.Medium
	default:
		return models.Critical
	}
}

func veendorInstall(sourcePath string) error {
	npmrcPath := path.Join(sourcePath, ".npmrc")
	if _, err := os.Stat(npmrcPath); err == nil {
		data, err := ioutil.ReadFile(npmrcPath)
		if err != nil {
			log.Printf("failed to read .npmrc %v\n", err)
			return err
		}
		re := regexp.MustCompile(`package-lock\s*=\s*false`)
		newData := re.ReplaceAllString(string(data), "package-lock=true")
		err = ioutil.WriteFile(npmrcPath, []byte(newData), 0)
		if err != nil {
			log.Printf("failed to replace package-lock in .npmrc %v\n", err)
			return err
		}
	} else if !os.IsNotExist(err) {
		log.Printf("failed to stat .npmrc %v\n", err)
		return err
	}

	cmd := exec.Command("veendor", "install", "-f")
	cmd.Dir = sourcePath
	cmd.Stdout = os.Stdout
	cmd.Stderr = os.Stderr
	err := cmd.Run()
	if err != nil {
		log.Printf("veendor install failed with error %v\n", err)
		return err
	}

	cmd = exec.Command("npm", "install", "--package-lock-only")
	cmd.Dir = sourcePath
	cmd.Stdout = os.Stdout
	cmd.Stderr = os.Stderr
	err = cmd.Run()
	if err != nil {
		log.Printf("npm install failed with error %v\n", err)
		return err
	}

	return nil
}

func isFilePresent(sourcePath, file string) bool {
	filePath := path.Join(sourcePath, file)
	_, err := os.Stat(filePath)
	return err == nil
}

func (r *Yadi) Init(sourcePath string, checkoutMeta *checkout.Checkout) error {
	// needed for golang SCA
	_ = os.Setenv("GOPATH", sourcePath)

	r.basePath = sourcePath
	r.checkoutMeta = checkoutMeta
	r.rawReport = make(map[string][]byte)

	return nil
}

func (r *Yadi) Run() error {
	for folder, info := range r.checkoutMeta.Folders {
		sourcePath := path.Join(r.basePath, folder)

		stat, err := os.Stat(sourcePath)
		if os.IsNotExist(err) {
			log.Printf("File or directory %s does not exists: %v\n", sourcePath, err)
			continue
		}

		useNonArcYadi := isFilePresent(sourcePath, "package.json") || isFilePresent(sourcePath, "maven_install.json")

		if *r.runVeendor != 0 {
			err := veendorInstall(sourcePath)
			if err != nil {
				log.Printf("veendorInstall failed with error %v\n", err)
			}
		}

		var cmd *exec.Cmd
		if info.IsArcadia && !useNonArcYadi {
			cmd = exec.Command("yadi-arc", "-f", "json", "--exit-status", "1", "--expand-groups", "test", sourcePath)
		} else {
			if stat.IsDir() {
				cmd = exec.Command("yadi", "-r", "-f", "json", "--exit-status", "1", "test", sourcePath)
			} else {
				cmd = exec.Command("yadi", "-f", "json", "--exit-status", "1", "test", sourcePath)
			}
		}
		if *r.feed != "" {
			cmd.Args = append(cmd.Args, []string{"--feed", *r.feed}...)
		}
		var stdout bytes.Buffer
		cmd.Stdout = &stdout
		cmd.Stderr = os.Stderr
		r.rawReport[folder] = nil

		if err := cmd.Run(); err != nil {
			if exiterr, ok := err.(*exec.ExitError); ok {
				if status, ok := exiterr.Sys().(syscall.WaitStatus); ok {
					exitStatus := status.ExitStatus()
					if exitStatus != 1 {
						log.Printf("Unexpected exit status \"%v\" for %v\n", exitStatus, cmd.Args)
					} else {
						r.rawReport[folder] = stdout.Bytes()
					}
				}
			}
		} else {
			r.rawReport[folder] = stdout.Bytes()
		}
	}
	return nil
}

func (r *Yadi) Normalize() error {
	if len(r.rawReport) == 0 {
		r.report = []*models.NewVulnerabilityDeduplicationRequestDTO{}
		return nil
	}

	r.report = make([]*models.NewVulnerabilityDeduplicationRequestDTO, 0)

	for folder, rawReport := range r.rawReport {
		stat, err := os.Stat(path.Join(r.basePath, folder))
		if os.IsNotExist(err) {
			log.Printf("File or directory %s does not exists: %v\n", folder, err)
			continue
		}

		var report []YadiReport
		err = json.Unmarshal(rawReport, &report)
		if err != nil {
			log.Println("Could not unmarshal yadi report:", err)
			continue
		}

		issuesMap := make(map[string]*models.NewVulnerabilityDeduplicationRequestDTO)

		for filePath := range report {
			relPath := strings.TrimPrefix(report[filePath].Path, path.Join(r.basePath, folder))
			for _, issue := range report[filePath].Issues {
				id := issue.ID
				packageName := issue.PackageName
				version := issue.Version
				var key string
				if r.checkoutMeta.Folders[folder].IsArcadia || !stat.IsDir() {
					key = fmt.Sprintf("%s:%s:%s", id, packageName, version)
				} else {
					key = fmt.Sprintf("%s:%s:%s:%s", relPath, id, packageName, version)
				}

				vulnerabilityPath := map[string]interface{}{
					"path":    issue.Path,
					"suggest": issue.Suggest,
				}

				dedupIssue, ok := issuesMap[key]
				if !ok {
					keyProperties := models.VulnerabilityProperties{
						"package_file": relPath,
						"id":           issue.ID,
						"version":      issue.Version,
						"package_name": issue.PackageName,
						"is_arcadia":   r.checkoutMeta.Folders[folder].IsArcadia,
					}
					fileURL, _ := checkout.GenerateFileURL(r.checkoutMeta.Folders[folder], relPath)
					if issue.Path == nil {
						issue.Path = []string{fmt.Sprintf("%s %s", issue.PackageName, issue.Version)}
					}
					if issue.Suggest == nil {
						issue.Suggest = []string{}
					}
					paths := append([]string{relPath}, issue.Path...)
					displayProperties := models.VulnerabilityProperties{
						"vulnerable_versions": issue.VulnerableVersions,
						"patched_versions":    issue.PatchedVersions,
						"summary":             issue.Summary,
						"reference":           issue.Reference,
						"cvss_score":          issue.CvssScore,
						"patch_exists":        issue.PatchExists,
						"paths":               [][]string{paths},
						"suggests":            [][]string{issue.Suggest},
						"vulnerability_paths": []map[string]interface{}{vulnerabilityPath},
						"file_url":            fileURL,
						"owners":              report[filePath].Owners,
					}
					category := issue.Summary
					if len(category) > 64 {
						category = "Unknown"
					}
					issuesMap[key] = &models.NewVulnerabilityDeduplicationRequestDTO{
						Severity:          cvssToSeverity(issue.CvssScore),
						Category:          category,
						KeyProperties:     keyProperties,
						DisplayProperties: displayProperties,
					}
				} else {
					dedupIssue.DisplayProperties["paths"] = append(dedupIssue.DisplayProperties["paths"].([][]string),
						append([]string{relPath}, issue.Path...))
					dedupIssue.DisplayProperties["suggests"] = append(dedupIssue.DisplayProperties["suggests"].([][]string), issue.Suggest)
					dedupIssue.DisplayProperties["vulnerability_paths"] = append(dedupIssue.DisplayProperties["vulnerability_paths"].([]map[string]interface{}),
						vulnerabilityPath)
				}
			}
		}

		for _, issue := range issuesMap {
			r.report = append(r.report, issue)
		}
	}

	return nil
}

func (r *Yadi) Type() models.ScanTypeName {
	return models.YADI
}

func (r *Yadi) RawReport() string {
	rawReport := "[\n"
	for _, rawSingleReport := range r.rawReport {
		rawReport += strings.TrimRight(string(rawSingleReport), "\n") + ",\n"
	}
	if len(rawReport) > 4 {
		rawReport = rawReport[:len(rawReport)-2]
	}
	rawReport += "]"

	return rawReport
}

func (r *Yadi) Report() interface{} {
	return r.report
}

func (r *Yadi) Version() string {
	return "0.1"
}
