package main

import (
	"bytes"
	"encoding/json"
	"log"
	"os"
	"os/exec"
	"path"
	"path/filepath"
	"strings"
	"syscall"

	"a.yandex-team.ru/security/impulse/models"
	"a.yandex-team.ru/security/impulse/workflow/internal/checkout"
	"a.yandex-team.ru/security/impulse/workflow/internal/dedup"
)

type YodaxIssue struct {
	Config      string `json:"config"`
	Description string `json:"description"`
	Path        string `json:"path"`
	Plugin      string `json:"plugin"`
	Reason      string `json:"reason"`
	Reference   string `json:"reference"`
	Severity    string `json:"severity"`
	Summary     string `json:"summary"`
}

type Yodax struct {
	basePath     string
	checkoutMeta *checkout.Checkout
	rawReport    map[string][][]byte
	report       []*models.NewVulnerabilityDeduplicationRequestDTO
}

var severityMap = map[string]models.SeverityType{
	"UNSPECIFIED": models.Info,
	"LOW":         models.Low,
	"MEDIUM":      models.Medium,
	"HIGH":        models.Critical,
}

func searchNginxConfigs(basePath string) ([]string, error) {
	if _, err := os.Stat(basePath); os.IsNotExist(err) {
		return nil, err
	}
	result := make([]string, 0)
	err := filepath.Walk(basePath, func(path string, info os.FileInfo, err error) error {
		if info.IsDir() {
			return nil
		}
		if strings.Contains(path, "/nginx/") && strings.HasSuffix(info.Name(), ".conf") || strings.HasSuffix(info.Name(), ".nginx") {
			result = append(result, path)
		}
		return nil
	})

	return result, err
}

func (r *Yodax) Init(sourcePath string, checkoutMeta *checkout.Checkout) error {
	r.basePath = sourcePath
	r.checkoutMeta = checkoutMeta
	r.rawReport = make(map[string][][]byte)

	return nil
}

func (r *Yodax) Run() error {
	for folder := range r.checkoutMeta.Folders {
		sourcePath := path.Join(r.basePath, folder)
		files, err := searchNginxConfigs(sourcePath)
		if err != nil {
			continue
		}
		r.rawReport[folder] = make([][]byte, 0)
		for _, file := range files {
			cmd := exec.Command("yodax", "-f", "json", file)
			var stdout bytes.Buffer
			cmd.Stdout = &stdout
			cmd.Stderr = os.Stderr

			if err := cmd.Run(); err != nil {
				if exiterr, ok := err.(*exec.ExitError); ok {
					if status, ok := exiterr.Sys().(syscall.WaitStatus); ok {
						exitStatus := status.ExitStatus()
						if exitStatus != 1 {
							log.Printf("Unexpected exit status \"%v\" for %v\n", exitStatus, cmd.Args)
						} else {
							r.rawReport[folder] = append(r.rawReport[folder], stdout.Bytes())
						}
					}
				}
			} else {
				r.rawReport[folder] = append(r.rawReport[folder], stdout.Bytes())
			}
		}
	}

	return nil
}

func (r *Yodax) Normalize() error {
	if len(r.rawReport) == 0 {
		r.report = []*models.NewVulnerabilityDeduplicationRequestDTO{}
		return nil
	}

	r.report = make([]*models.NewVulnerabilityDeduplicationRequestDTO, 0)
	for folder, rawRepositoryReport := range r.rawReport {
		report := make([]*YodaxIssue, 0)
		for _, rawSingleReport := range rawRepositoryReport {
			var issues []*YodaxIssue
			err := json.Unmarshal(rawSingleReport, &issues)
			if err != nil {
				log.Println("Could not unmarshal yodax report:", err)
				continue
			}
			report = append(report, issues...)
		}

		for _, issue := range report {
			filePath := strings.TrimPrefix(issue.Path, path.Join(r.basePath, folder))
			fileURL, _ := checkout.GenerateFileURL(r.checkoutMeta.Folders[folder], filePath)
			configHash, _ := dedup.GetStringHash(issue.Config)

			keyProperties := models.VulnerabilityProperties{
				"filename":    filePath,
				"config_hash": configHash,
				"plugin":      issue.Plugin,
			}
			displayProperties := models.VulnerabilityProperties{
				"config":      issue.Config,
				"description": issue.Description,
				"reference":   issue.Reference,
				"severity":    severityMap[issue.Severity],
				"summary":     issue.Summary,
				"reason":      issue.Reason,
				"file_url":    fileURL,
			}
			r.report = append(r.report, &models.NewVulnerabilityDeduplicationRequestDTO{
				Severity:          severityMap[issue.Severity],
				Category:          issue.Plugin,
				KeyProperties:     keyProperties,
				DisplayProperties: displayProperties,
			})
		}
	}

	return nil
}

func (r *Yodax) Type() models.ScanTypeName {
	return models.YODAX
}

func (r *Yodax) RawReport() string {
	rawReport := "[\n"
	for _, rawRepositoryReport := range r.rawReport {
		for _, rawSingleReport := range rawRepositoryReport {
			rawReport += strings.TrimRight(string(rawSingleReport), "\n") + ",\n"
		}
	}
	if len(rawReport) > 4 {
		rawReport = rawReport[:len(rawReport)-2]
	}
	rawReport += "]"

	return rawReport
}

func (r *Yodax) Report() interface{} {
	return r.report
}

func (r *Yodax) Version() string {
	return "0.1"
}
