package main

import (
	"context"
	"flag"
	"fmt"
	"os"
	"os/signal"
	"syscall"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/core/log/zap"
	"a.yandex-team.ru/library/go/maxprocs"
	"a.yandex-team.ru/security/kirby/internal/app"
	"a.yandex-team.ru/security/kirby/internal/config"
)

func logger(logLevel string) log.Logger {
	lvl, err := log.ParseLevel(logLevel)
	if err != nil {
		_, _ = fmt.Fprintf(os.Stderr, "failed to parse log lvl, Info used: %v\n", err)
		lvl = log.InfoLevel
	}

	l, err := zap.NewDeployLogger(lvl)
	if err != nil {
		panic(fmt.Sprintf("can't create logger: %v", err))
	}

	return l
}

func main() {
	var (
		cfgPath string
	)
	flag.StringVar(&cfgPath, "cfg", "", "config path")
	flag.Parse()

	maxprocs.AdjustAuto()
	cfg, err := config.NewConfig(cfgPath)
	if err != nil {
		panic(fmt.Sprintf("can't parse config: %v", err))
	}

	l := logger(cfg.LogLevel)

	instance, err := app.NewApp(l, cfg)
	if err != nil {
		l.Fatal("can't create application instance", log.Error(err))
		return
	}

	errChan := make(chan error, 1)
	okChan := make(chan struct{}, 1)
	go func() {
		if err := instance.Start(); err != nil {
			errChan <- err
		} else {
			okChan <- struct{}{}
		}
	}()

	stopChan := make(chan os.Signal, 2)
	signal.Notify(stopChan, syscall.SIGINT, syscall.SIGTERM)

	select {
	case <-stopChan:
		l.Info("shutdown signal received")

		ctx, cancel := context.WithTimeout(context.Background(), config.ShutdownDeadline)
		defer cancel()

		err := instance.Shutdown(ctx)
		if err != nil {
			l.Error("failed to gracefully shutdown Kirby", log.Error(err))
		}
	case <-okChan:
	case err := <-errChan:
		l.Error("failed to start Kirby", log.Error(err))
	}
}
