package app

import (
	"context"
	"fmt"
	"net/http"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/security/kirby/internal/config"
	"a.yandex-team.ru/security/kirby/internal/gopher"
	"a.yandex-team.ru/security/kirby/internal/storage"
)

type App struct {
	log     log.Logger
	cfg     *config.Config
	gopher  *gopher.Gopher
	storage storage.Storage
	http    *http.Server
}

func NewApp(l log.Logger, cfg *config.Config) (*App, error) {
	return &App{
		log: l,
		cfg: cfg,
	}, nil
}

func (a *App) initialize() error {
	var err error
	switch a.cfg.Storage.Kind {
	case config.StorageKindS3:
		a.storage, err = storage.NewS3Storage(a.cfg.Storage.S3)
	case config.StorageKingDisk:
		a.storage, err = storage.NewDiskStorage(a.cfg.Storage.Disk)
	default:
		err = fmt.Errorf("unknown storage kind: %s", a.cfg.Storage.Kind)
	}

	if err != nil {
		return fmt.Errorf("failed to create storage: %w", err)
	}

	a.gopher, err = gopher.NewGopher(a.cfg.Gopher, gopher.WithLogger(a.log), gopher.WithStorage(a.storage))
	if err != nil {
		return fmt.Errorf("failed to create gopher: %w", err)
	}

	a.http = &http.Server{
		Addr:    fmt.Sprintf(":%d", a.cfg.HTTPPort),
		Handler: newRouter(a),
	}
	return nil
}

func (a *App) Start() error {
	if err := a.initialize(); err != nil {
		return fmt.Errorf("initialization failed: %w", err)
	}

	a.log.Info("HTTP listen on " + a.http.Addr)
	return a.http.ListenAndServe()
}

func (a *App) Shutdown(ctx context.Context) error {
	// not initialized yet
	if a.http == nil {
		return nil
	}

	err := a.http.Shutdown(ctx)
	if ctx.Err() != nil {
		return ctx.Err()
	}

	if err != nil {
		a.log.Error("failed to shutdown HTTP server", log.Error(err))
	}
	return a.gopher.Shutdown(ctx)
}
