package app

import (
	"errors"
	"net/http"

	"a.yandex-team.ru/security/kirby/internal/module"
)

const (
	contentTypeJSON = "application/json"
	contentTypeText = "text/plain; charset=UTF-8"
)

// protocol RFC: https://golang.org/cmd/go/#hdr-Module_proxy_protocol
func (a *App) modHandler(w http.ResponseWriter, r *http.Request) {
	m, err := module.ParseModule(r.URL.Path)
	if err != nil {
		errResponse(w, r, err, http.StatusBadRequest)
		return
	}

	err = module.ValidateModule(m)
	if err != nil {
		errResponse(w, r, err, http.StatusGone)
		return
	}

	switch m.What {
	case module.WhatLatest:
		err = a.doModLatest(w, r, m)
	case module.WhatList:
		err = a.doModList(w, r, m)
	default:
		err = a.doModDownload(w, r, m)
	}

	if err != nil {
		errResponse(w, r, err, http.StatusGone)
		return
	}
}

func (a *App) doModLatest(w http.ResponseWriter, r *http.Request, m *module.Module) error {
	setResponseContentType(w, contentTypeJSON)
	setResponseCacheControlHeader(w, 1800)

	latestBody, err := a.gopher.Latest(r.Context(), m)
	if err != nil {
		return err
	}

	_, _ = w.Write(latestBody)
	return nil
}

func (a *App) doModList(w http.ResponseWriter, r *http.Request, m *module.Module) error {
	setResponseContentType(w, contentTypeText)
	setResponseCacheControlHeader(w, 1800)

	listBody, err := a.gopher.List(r.Context(), m)
	if err != nil {
		return err
	}

	_, _ = w.Write(listBody)
	return nil
}

func (a *App) doModDownload(w http.ResponseWriter, r *http.Request, m *module.Module) error {
	switch m.What {
	case module.WhatInfo:
		setResponseContentType(w, contentTypeJSON)
		setResponseCacheControlHeader(w, 86400)

		modInfo, err := a.gopher.Info(r.Context(), m)
		if err != nil {
			return err
		}

		_, _ = w.Write(modInfo)
		return nil
	case module.WhatMod:
		setResponseContentType(w, contentTypeText)
		setResponseCacheControlHeader(w, 86400)

		modContent, err := a.gopher.Mod(r.Context(), m)
		if err != nil {
			return err
		}

		_, _ = w.Write(modContent)
		return err
	case module.WhatZip:
		err := a.gopher.Zip(r.Context(), m)
		if err != nil {
			return err
		}

		if a.cfg.DirectDownload {
			return a.storage.WriteTo(r.Context(), m.ZipPath(), w)
		}
		http.Redirect(w, r, a.storage.DownloadURI(m.ZipPath()), http.StatusMovedPermanently)
		return nil
	default:
		return errors.New("unknown module request type")
	}
}
