package config

import (
	"context"
	"fmt"
	"time"

	"github.com/heetch/confita"
	"github.com/heetch/confita/backend"
	"github.com/heetch/confita/backend/env"
	"github.com/heetch/confita/backend/file"

	"a.yandex-team.ru/library/go/core/log"
)

const (
	ShutdownDeadline = 5 * time.Minute
	StorageKindS3    = "s3"
	StorageKingDisk  = "disk"
)

type StorageKind string

type GoBin struct {
	Enabled      bool          `config:"gobin_enabled" yaml:"enabled"`
	Timeout      time.Duration `config:"gobin_timeout" yaml:"timeout"`
	MaxDiskUsage uint64        `config:"gobin_max_disk_usage" yaml:"max_disk_usage"`
	MaxWorkers   int           `config:"gobin_max_workers" yaml:"max_workers"`
	GoPath       string        `config:"gobin_go_path" yaml:"go_path"`
}

type GoProxy struct {
	Enabled  bool          `config:"goproxy_enabled" yaml:"enabled"`
	Endpoint string        `config:"goproxy_endpoint" yaml:"endpoint"`
	Timeout  time.Duration `config:"goproxy_timeout" yaml:"timeout"`
}

type Gopher struct {
	ListTTL time.Duration `config:"list_ttl" yaml:"list_ttl"`
	GoBin   GoBin         `config:"gobin" yaml:"gobin"`
	GoProxy GoProxy       `config:"goproxy" yaml:"goproxy"`
}

type S3Storage struct {
	MaxRetries      int           `config:"s3_max_retries" yaml:"max_retries"`
	UploadTimeout   time.Duration `config:"s3_upload_timeout" yaml:"upload_timeout"`
	Endpoint        string        `config:"s3_endpoint" yaml:"endpoint"`
	Bucket          string        `config:"s3_bucket" yaml:"bucket"`
	AccessKeyID     string        `config:"s3_access_key_id" yaml:"access_key_id"`
	AccessSecretKey string        `config:"s3_access_secret_key" yaml:"access_secret_key"`
}

type DiskStorage struct {
	Dir string `config:"disk_dir" yaml:"dir"`
}

type Storage struct {
	Kind StorageKind `config:"storage_kind" yaml:"kind"`
	S3   S3Storage   `config:"s3" yaml:"s3"`
	Disk DiskStorage `config:"disk" yaml:"disk"`
}

type Config struct {
	LogLevel       string  `config:"log_level" yaml:"log_level"`
	HTTPPort       int     `config:"http_port" yaml:"http_port"`
	DirectDownload bool    `config:"direct_download" yaml:"direct_download"`
	Storage        Storage `config:"storage" yaml:"storage"`
	Gopher         Gopher  `config:"gopher" yaml:"gopher"`
}

func NewConfig(configPath string) (*Config, error) {
	cfg := &Config{
		LogLevel:       log.InfoString,
		HTTPPort:       3000,
		DirectDownload: false,
		Storage: Storage{
			Kind: StorageKindS3,
			S3: S3Storage{
				MaxRetries:    2,
				UploadTimeout: 5 * time.Minute,
				Endpoint:      "s3.mdst.yandex.net",
				Bucket:        "kirby",
			},
		},
		Gopher: Gopher{
			ListTTL: 30 * time.Minute,
			GoBin: GoBin{
				Enabled:      true,
				Timeout:      5 * time.Minute,
				MaxDiskUsage: 70,
				MaxWorkers:   10,
				GoPath:       "/tmp/go",
			},
			GoProxy: GoProxy{
				Enabled:  true,
				Timeout:  2 * time.Minute,
				Endpoint: "http://proxy.golang.org",
			},
		},
	}

	backends := []backend.Backend{env.NewBackend()}
	if configPath != "" {
		backends = append(
			[]backend.Backend{file.NewBackend(configPath)},
			backends...,
		)
	}

	if err := confita.NewLoader(backends...).Load(context.Background(), cfg); err != nil {
		return nil, fmt.Errorf("failed to parse config: %w", err)
	}

	return cfg, nil
}
