package gopher

import (
	"bytes"
	"context"
	"encoding/json"
	"errors"
	"fmt"
	"io"
	"io/ioutil"
	"sync"
	"time"

	"github.com/karlseguin/ccache/v2"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/core/log/ctxlog"
	"a.yandex-team.ru/library/go/core/log/nop"
	"a.yandex-team.ru/security/kirby/internal/config"
	"a.yandex-team.ru/security/kirby/internal/gopher/resolver/gobin"
	"a.yandex-team.ru/security/kirby/internal/gopher/resolver/goproxy"
	"a.yandex-team.ru/security/kirby/internal/gopher/resolver/resolver"
	"a.yandex-team.ru/security/kirby/internal/module"
	"a.yandex-team.ru/security/kirby/internal/storage"
)

const (
	contentTypeJSON  = "application/json"
	contentTypeText  = "text/plain; charset=UTF-8"
	contentTypeOctet = "application/octet-stream"

	defaultCacheTTL = 24 * time.Hour
)

var (
	ErrNoResolvers = errors.New("no resolvers configured")
	ErrClosed      = errors.New("closed")
	ErrExpired     = errors.New("expired")
)

type Gopher struct {
	l         log.Logger
	storage   storage.Storage
	resolvers []resolver.Resolver
	closeMu   sync.RWMutex
	closed    bool
	listTTL   time.Duration
	cache     *ccache.Cache
}

func NewGopher(cfg config.Gopher, opts ...Option) (*Gopher, error) {
	g := &Gopher{
		l:       &nop.Logger{},
		listTTL: cfg.ListTTL,
		cache:   ccache.New(ccache.Configure().MaxSize(2048)),
	}

	for _, opt := range opts {
		opt(g)
	}

	if g.storage == nil {
		return nil, errors.New("no storage configured")
	}

	if cfg.GoProxy.Enabled {
		r, err := goproxy.NewGoProxy(g.l, cfg.GoProxy)
		if err != nil {
			return nil, fmt.Errorf("failed to create goproxy resolver: %w", err)
		}
		g.resolvers = append(g.resolvers, r)
	}

	if cfg.GoBin.Enabled {
		r, err := gobin.NewGoBin(g.l, cfg.GoBin)
		if err != nil {
			return nil, fmt.Errorf("failed to create gobin resolver: %w", err)
		}
		g.resolvers = append(g.resolvers, r)
	}

	if len(g.resolvers) == 0 {
		return nil, errors.New("no resolvers configured")
	}

	return g, nil
}

func (g *Gopher) List(ctx context.Context, m *module.Module) ([]byte, error) {
	cacheKey := m.ListPath()
	if out, err := g.fromCache(ctx, cacheKey, true); err == nil {
		return out, err
	}

	var lastErr error
	for _, r := range g.resolvers {
		out, err := r.List(ctx, m)
		if err != nil {
			lastErr = err
			ctxlog.Warn(ctx, g.l,
				"failed to list module",
				log.String("resolver", r.Name()),
				log.String("module", m.String()),
				log.Error(err),
			)
			continue
		}

		return out, g.saveBytes(cacheKey, contentTypeText, out)
	}

	if lastErr != nil {
		return nil, lastErr
	}

	return nil, ErrNoResolvers
}

func (g *Gopher) Latest(ctx context.Context, m *module.Module) ([]byte, error) {
	cacheKey := m.LatestPath()
	if out, err := g.fromCache(ctx, cacheKey, true); err == nil {
		return out, err
	}

	var lastErr error
	for _, r := range g.resolvers {
		latest, err := r.Latest(ctx, m)
		if err != nil {
			lastErr = err
			ctxlog.Warn(ctx, g.l,
				"failed to get latest module info",
				log.String("resolver", r.Name()),
				log.String("module", m.String()),
				log.Error(err),
			)
			continue
		}

		return g.saveJSON(cacheKey, latest)
	}

	if lastErr != nil {
		return nil, lastErr
	}

	return nil, ErrNoResolvers
}

func (g *Gopher) Info(ctx context.Context, m *module.Module) ([]byte, error) {
	cacheKey := m.InfoPath()
	if out, err := g.fromCache(ctx, cacheKey, false); err == nil {
		return out, err
	}

	var lastErr error
	for _, r := range g.resolvers {
		modInfo, err := r.Info(ctx, m)
		if err != nil {
			lastErr = err
			ctxlog.Warn(ctx, g.l,
				"failed to get module info",
				log.String("resolver", r.Name()),
				log.String("module", m.String()),
				log.Error(err),
			)
			continue
		}

		return g.saveJSON(cacheKey, modInfo)
	}

	if lastErr != nil {
		return nil, lastErr
	}

	return nil, ErrNoResolvers
}

func (g *Gopher) Mod(ctx context.Context, m *module.Module) ([]byte, error) {
	if out, err := g.fromCache(ctx, m.ModPath(), false); err == nil {
		return out, err
	}

	var lastErr error
	for _, r := range g.resolvers {
		goMod, _, err := g.downloadModule(ctx, m, r)
		if err != nil {
			lastErr = err
			ctxlog.Warn(ctx, g.l,
				"failed to download module",
				log.String("resolver", r.Name()),
				log.String("module", m.String()),
				log.Error(err),
			)
			continue
		}

		return goMod, nil
	}

	if lastErr != nil {
		return nil, lastErr
	}

	return nil, ErrNoResolvers
}

func (g *Gopher) Zip(ctx context.Context, m *module.Module) error {
	if _, err := g.storage.Head(ctx, m.ZipPath()); err == nil {
		return nil
	}

	var lastErr error
	for _, r := range g.resolvers {
		_, _, err := g.downloadModule(ctx, m, r)
		if err != nil {
			lastErr = err
			ctxlog.Warn(ctx, g.l,
				"failed to download module",
				log.String("resolver", r.Name()),
				log.String("module", m.String()),
				log.Error(err),
			)
			continue
		}

		return nil
	}

	if lastErr != nil {
		return lastErr
	}

	return ErrNoResolvers
}

func (g *Gopher) Shutdown(ctx context.Context) error {
	if g.closed {
		return ErrClosed
	}

	defer g.cache.Stop()

	done := make(chan struct{})
	go func() {
		g.closeMu.Lock()
		g.closed = true
		g.closeMu.Unlock()
		close(done)
	}()

	select {
	case <-ctx.Done():
		return ctx.Err()
	case <-done:
		return nil
	}
}

func (g *Gopher) fromCache(ctx context.Context, key string, isList bool) ([]byte, error) {
	if item := g.cache.Get(key); item != nil {
		if !isList || !item.Expired() {
			return item.Value().([]byte), nil
		}
	}

	obj, err := g.storage.Get(ctx, key)
	if err != nil {
		return nil, err
	}

	if isList && time.Since(obj.UpdatedAt) > g.listTTL {
		return nil, ErrExpired
	}

	out, err := ioutil.ReadAll(obj.Body)
	_ = obj.Body.Close()
	if err == nil {
		ttl := defaultCacheTTL
		if isList {
			ttl = time.Until(obj.UpdatedAt.Add(g.listTTL))
		}

		g.cache.Set(key, out, ttl)
	}

	return out, err
}

func (g *Gopher) downloadModule(ctx context.Context, m *module.Module, r resolver.Resolver) ([]byte, []byte, error) {
	// go request file in order: info, mod, zip
	// we save in reverse: zip, mod, info

	dInfo, err := r.Download(ctx, m)
	if err != nil {
		return nil, nil, err
	}

	// we protect whole module save with lock
	g.closeMu.RLock()
	defer g.closeMu.RUnlock()

	if err := g.save(m.ZipPath(), contentTypeOctet, dInfo.Zip); err != nil {
		_ = dInfo.Zip.Close()
		return nil, nil, fmt.Errorf("failed to save module zip: %w", err)
	}
	_ = dInfo.Zip.Close()

	if err := g.saveBytes(m.ModPath(), contentTypeText, dInfo.Mod); err != nil {
		return nil, nil, fmt.Errorf("failed to save module mod: %w", err)
	}

	if err := g.saveBytes(m.InfoPath(), contentTypeJSON, dInfo.Info); err != nil {
		return nil, nil, fmt.Errorf("failed to save module info: %w", err)
	}

	return dInfo.Mod, dInfo.Info, nil
}

func (g *Gopher) save(path string, ct string, data io.Reader) error {
	g.closeMu.RLock()
	defer g.closeMu.RUnlock()

	if g.closed {
		return ErrClosed
	}

	// we don't use user context to prevent upload interrupt
	return g.storage.Put(context.Background(), path, storage.PutItem{
		Body:        data,
		ContentType: ct,
	})
}

func (g *Gopher) saveBytes(path string, ct string, data []byte) error {
	return g.save(path, ct, bytes.NewReader(data))
}

func (g *Gopher) saveJSON(path string, data interface{}) ([]byte, error) {
	body, err := json.Marshal(data)
	if err != nil {
		return nil, fmt.Errorf("failed to marshal: %w", err)
	}

	return body, g.saveBytes(path, contentTypeJSON, body)
}
