package gobin

import (
	"bytes"
	"context"
	"encoding/json"
	"fmt"
	"io/ioutil"
	"os"
	"os/exec"
	"strings"
	"sync"
	"syscall"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/security/kirby/internal/config"
	"a.yandex-team.ru/security/kirby/internal/gopher/resolver/resolver"
	"a.yandex-team.ru/security/kirby/internal/module"
)

var _ resolver.Resolver = (*GoBin)(nil)

type (
	GoBin struct {
		l          log.Logger
		env        []string
		workerChan chan struct{}
		freeMu     sync.RWMutex
		cfg        config.GoBin
	}

	goDownloadResult struct {
		Version string
		Info    string
		GoMod   string
		Zip     string
	}

	goListResult struct {
		Versions []string
	}

	goError struct {
		Error string
	}
)

func NewGoBin(l log.Logger, cfg config.GoBin) (*GoBin, error) {
	if err := CheckRequirements(); err != nil {
		return nil, err
	}

	if cfg.MaxWorkers <= 0 {
		return nil, fmt.Errorf("max workers must be gather than 0, current: %d", cfg.MaxWorkers)
	}

	goPrivate := "*.yandex-team.ru"
	if envPrivate := os.Getenv("GOPRIVATE"); envPrivate != "" {
		goPrivate += "," + envPrivate
	}

	env := append(
		os.Environ(),
		"GO111MODULE=on",
		"GOPROXY=direct",
		"GONOPROXY=",
		"GOSUMDB=off",
		"GONOSUMDB=",
		"GOPRIVATE="+goPrivate,
	)

	if cfg.GoPath != "" {
		if err := os.MkdirAll(cfg.GoPath, 0755); err != nil {
			return nil, fmt.Errorf("failed to prepare gopath: %w", err)
		}

		env = append(env, "GOPATH="+cfg.GoPath)
	}

	return &GoBin{
		l:          l,
		env:        env,
		workerChan: make(chan struct{}, cfg.MaxWorkers),
		cfg:        cfg,
	}, nil
}

func (g *GoBin) Name() string {
	return "gobin"
}

func (g *GoBin) List(ctx context.Context, m *module.Module) ([]byte, error) {
	var lr goListResult
	err := g.goJSON(ctx, &lr, "list", "-json", "-mod=readonly", "-versions", "-m", m.Path+"@latest")
	if err != nil {
		return nil, err
	}

	var out bytes.Buffer
	for _, v := range lr.Versions {
		_, _ = out.WriteString(v)
		_ = out.WriteByte('\n')
	}

	return out.Bytes(), nil
}

func (g *GoBin) Latest(ctx context.Context, m *module.Module) (resolver.InfoResult, error) {
	var out resolver.InfoResult
	return out, g.goJSON(ctx, &out, "list", "-json", "-mod=readonly", "-versions", "-m", m.Path+"@latest")
}

func (g *GoBin) Info(ctx context.Context, m *module.Module) (resolver.InfoResult, error) {
	var out resolver.InfoResult
	return out, g.goJSON(ctx, &out, "list", "-json", "-mod=readonly", "-m", m.Path+"@"+m.Version)
}

func (g *GoBin) Download(ctx context.Context, m *module.Module) (resolver.DownloadResult, error) {
	var out resolver.DownloadResult
	var dInfo goDownloadResult
	err := g.goJSON(ctx, &dInfo, "mod", "download", "-json", m.Path+"@"+m.Version)
	if err != nil {
		return out, fmt.Errorf("failed to download module: %w", err)
	}

	out.Info, err = ioutil.ReadFile(dInfo.Info)
	if err != nil {
		return out, fmt.Errorf("failed to read module info: %w", err)
	}

	out.Mod, err = ioutil.ReadFile(dInfo.GoMod)
	if err != nil {
		return out, fmt.Errorf("failed to read module mod: %w", err)
	}

	out.Zip, err = os.Open(dInfo.Zip)
	if err != nil {
		return out, fmt.Errorf("failed to open module zip: %w", err)
	}
	return out, nil
}

func (g *GoBin) goJSON(ctx context.Context, dst interface{}, args ...string) error {
	select {
	case g.workerChan <- struct{}{}:
		defer func() { <-g.workerChan }()
	case <-ctx.Done():
		return ctx.Err()
	}

	g.checkDiskUsage()
	g.freeMu.RLock()
	defer g.freeMu.RUnlock()

	ctx, cancel := context.WithTimeout(ctx, g.cfg.Timeout)
	defer cancel()

	cmd := exec.CommandContext(ctx, "go", args...)
	var stdout, stderr bytes.Buffer
	cmd.Env = g.env
	cmd.Stdout = &stdout
	cmd.Stderr = &stderr
	if err := cmd.Run(); err != nil {
		var goErr goError
		if jErr := json.Unmarshal(stdout.Bytes(), &goErr); jErr == nil && goErr.Error != "" {
			return fmt.Errorf("go %s: %v: %s", strings.Join(args, " "), err, goErr.Error)
		}

		return fmt.Errorf("go %s: %v\n%s", strings.Join(args, " "), err, stderr.String())
	}

	if err := json.Unmarshal(stdout.Bytes(), dst); err != nil {
		return fmt.Errorf("go %s: reading json: %v", strings.Join(args, " "), err)
	}
	return nil
}

func (g *GoBin) goRun(args ...string) error {
	cmd := exec.Command("go", args...)
	var stdout, stderr bytes.Buffer
	cmd.Env = g.env
	cmd.Stdout = &stdout
	cmd.Stderr = &stderr
	if err := cmd.Run(); err != nil {
		return fmt.Errorf("go %s: %v\n%s", strings.Join(args, " "), err, stderr.String())
	}

	return nil
}

func (g *GoBin) checkDiskUsage() {
	if g.cfg.GoPath == "" || g.cfg.MaxDiskUsage == 0 {
		return
	}

	checkSpace := func() bool {
		var stat syscall.Statfs_t
		err := syscall.Statfs(g.cfg.GoPath, &stat)
		if err != nil {
			g.l.Error("failed to statfs", log.String("path", g.cfg.GoPath), log.Error(err))
			return true
		}

		if (stat.Blocks-stat.Bavail)*100/stat.Blocks < g.cfg.MaxDiskUsage {
			return true
		}
		return false
	}

	if checkSpace() {
		return
	}

	g.freeMu.Lock()
	defer g.freeMu.Unlock()

	// check space second time to deal with multiple writers
	if checkSpace() {
		return
	}

	err := g.goRun("clean", "-cache", "-modcache")
	if err != nil {
		g.l.Error("failed to clean go cache", log.Error(err))
	}
}
