package goproxy

import (
	"context"
	"encoding/json"
	"errors"
	"fmt"
	"io"

	"github.com/go-resty/resty/v2"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/security/kirby/internal/config"
	"a.yandex-team.ru/security/kirby/internal/gopher/resolver/resolver"
	"a.yandex-team.ru/security/kirby/internal/module"
)

var _ resolver.Resolver = (*GoProxy)(nil)

type GoProxy struct {
	l     log.Logger
	httpc *resty.Client
}

func NewGoProxy(l log.Logger, cfg config.GoProxy) (*GoProxy, error) {
	return &GoProxy{
		l: l,
		httpc: resty.New().
			SetBaseURL(cfg.Endpoint).
			SetRedirectPolicy(resty.FlexibleRedirectPolicy(5)).
			SetTimeout(cfg.Timeout),
	}, nil
}

func (g *GoProxy) Name() string {
	return "goproxy"
}

func (g *GoProxy) List(ctx context.Context, m *module.Module) ([]byte, error) {
	rsp, err := g.httpc.R().
		SetContext(ctx).
		Get(m.ListPath())
	if err != nil {
		return nil, err
	}

	if !rsp.IsSuccess() {
		return nil, fmt.Errorf("non-200 status from upstream: %s", rsp.Status())
	}

	return rsp.Body(), nil
}

func (g *GoProxy) Latest(ctx context.Context, m *module.Module) (resolver.InfoResult, error) {
	var out resolver.InfoResult
	rsp, err := g.httpc.R().
		SetContext(ctx).
		SetResult(&out).
		Get(m.LatestPath())
	if err != nil {
		return out, err
	}

	if !rsp.IsSuccess() {
		return out, fmt.Errorf("non-200 status from upstream: %s", rsp.Status())
	}

	if out.Version == "" {
		return out, errors.New("invalid proxy response: no version")
	}

	return out, nil
}

func (g *GoProxy) Info(ctx context.Context, m *module.Module) (resolver.InfoResult, error) {
	var out resolver.InfoResult
	rsp, err := g.httpc.R().
		SetContext(ctx).
		SetResult(&out).
		Get(m.InfoPath())
	if err != nil {
		return out, err
	}

	if !rsp.IsSuccess() {
		return out, fmt.Errorf("non-200 status from upstream: %s", rsp.Status())
	}

	if out.Version == "" {
		return out, errors.New("invalid proxy response: no version")
	}

	return out, nil
}

func (g *GoProxy) Mod(ctx context.Context, m *module.Module) ([]byte, error) {
	rsp, err := g.httpc.R().
		SetContext(ctx).
		Get(m.ModPath())
	if err != nil {
		return nil, err
	}

	if !rsp.IsSuccess() {
		return nil, fmt.Errorf("non-200 status from upstream: %s", rsp.Status())
	}

	modContent := rsp.Body()
	if len(modContent) == 0 {
		return nil, errors.New("invalid proxy response: no content")
	}

	return modContent, nil
}

func (g *GoProxy) Zip(ctx context.Context, m *module.Module) (io.ReadCloser, error) {
	rsp, err := g.httpc.R().
		SetContext(ctx).
		SetDoNotParseResponse(true).
		Get(m.ZipPath())
	if err != nil {
		return nil, err
	}

	if !rsp.IsSuccess() {
		return nil, fmt.Errorf("non-200 status from upstream: %s", rsp.Status())
	}

	return rsp.RawBody(), nil
}

func (g *GoProxy) Download(ctx context.Context, m *module.Module) (resolver.DownloadResult, error) {
	var out resolver.DownloadResult
	goInfo, err := g.Info(ctx, m)
	if err != nil {
		return out, fmt.Errorf("failed to request module info: %w", err)
	}

	out.Info, err = json.Marshal(goInfo)
	if err != nil {
		return out, fmt.Errorf("failed to marshal module info: %w", err)
	}

	out.Mod, err = g.Mod(ctx, m)
	if err != nil {
		return out, fmt.Errorf("failed to request module mod: %w", err)
	}

	out.Zip, err = g.Zip(ctx, m)
	if err != nil {
		return out, fmt.Errorf("failed to request module zip: %w", err)
	}

	return out, nil
}
