package middleware

import (
	"context"
	"fmt"
	"net/http"
	"os"
	"sync/atomic"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/core/log/ctxlog"
)

const (
	RequestIDHeader = "X-Request-ID"
)

var (
	prefix = func() string {
		hostname, err := os.Hostname()
		if hostname == "" || err != nil {
			hostname = "localhost"
		}
		return hostname + "/"
	}()

	reqid        uint64
	requestIDKey = &ctxKey{}
)

type ctxKey struct{}

func RequestID(next http.Handler) http.Handler {
	fn := func(w http.ResponseWriter, r *http.Request) {
		ctx := r.Context()
		requestID := r.Header.Get(RequestIDHeader)
		if requestID == "" {
			curID := atomic.AddUint64(&reqid, 1)
			requestID = fmt.Sprintf("%06d", curID)
		}

		requestID = prefix + requestID
		ctx = context.WithValue(ctx, requestIDKey, requestID)
		ctx = ctxlog.WithFields(ctx, log.String("request_id", requestID))
		next.ServeHTTP(w, r.WithContext(ctx))
	}
	return http.HandlerFunc(fn)
}

func GetReqID(ctx context.Context) string {
	if ctx == nil {
		return ""
	}

	if reqID, ok := ctx.Value(requestIDKey).(string); ok {
		return reqID
	}
	return ""
}
