package module

import (
	"fmt"
	"path"
	"strings"

	"golang.org/x/mod/module"
)

const (
	WhatList What = iota
	WhatLatest
	WhatInfo
	WhatMod
	WhatZip
)

type (
	What uint8
)

type Module struct {
	Path           string
	What           What
	Version        string
	EscapedPath    string
	EscapedVersion string
}

func ParseModule(uriPath string) (*Module, error) {
	pathIdx := strings.LastIndex(uriPath, "/@")
	if pathIdx <= 0 {
		return nil, ErrInvalidModulePath
	}

	rawPath, what := uriPath[:pathIdx], uriPath[pathIdx+2:]
	modPath, err := module.UnescapePath(strings.TrimPrefix(rawPath, "/"))
	if err != nil {
		return nil, fmt.Errorf("invalid module path: %w", err)
	}

	escapedModPath, err := module.EscapePath(modPath)
	if err != nil {
		return nil, fmt.Errorf("invalid module path: %w", err)
	}

	m := Module{
		Path:        modPath,
		EscapedPath: escapedModPath,
	}
	switch what {
	case "latest":
		m.What = WhatLatest
	case "v/list":
		m.What = WhatList
	default:
		if !strings.HasPrefix(what, "v/") {
			return nil, ErrInvalidModulePath
		}

		ext := path.Ext(what)
		m.Version, err = module.UnescapeVersion(strings.TrimSuffix(what[2:], ext))
		if err != nil {
			return nil, fmt.Errorf("invalid version module '%s' version: %w", modPath, err)
		}

		m.EscapedVersion, err = module.EscapeVersion(m.Version)
		if err != nil {
			return nil, fmt.Errorf("invalid version module '%s' version: %w", modPath, err)
		}

		switch ext {
		case ".info":
			m.What = WhatInfo
		case ".mod":
			m.What = WhatMod
		case ".zip":
			m.What = WhatZip
		default:
			return nil, fmt.Errorf("unknown module request type: %s", ext)
		}
	}

	return &m, nil
}

func (m *Module) LatestPath() string {
	return m.EscapedPath + "/@latest"
}

func (m *Module) ListPath() string {
	return m.EscapedPath + "/@v/list"
}

func (m *Module) InfoPath() string {
	return fmt.Sprintf("%s/@v/%s.info", m.EscapedPath, m.EscapedVersion)
}

func (m *Module) ModPath() string {
	return fmt.Sprintf("%s/@v/%s.mod", m.EscapedPath, m.EscapedVersion)
}

func (m *Module) ZipPath() string {
	return fmt.Sprintf("%s/@v/%s.zip", m.EscapedPath, m.EscapedVersion)
}

func (m *Module) String() string {
	switch m.What {
	case WhatList, WhatLatest:
		return m.Path
	default:
		return fmt.Sprintf("%s@%s", m.Path, m.Version)
	}
}
