package storage

import (
	"context"
	"fmt"
	"io"
	"io/ioutil"
	"os"
	"path"
	"path/filepath"

	"a.yandex-team.ru/security/kirby/internal/config"
)

var _ Storage = (*DiskStorage)(nil)

type DiskStorage struct {
	root string
}

func NewDiskStorage(cfg config.DiskStorage) (*DiskStorage, error) {
	if err := os.MkdirAll(cfg.Dir, 0755); err != nil {
		return nil, fmt.Errorf("failed to create root dir: %w", err)
	}

	return &DiskStorage{root: cfg.Dir}, nil
}

func (s *DiskStorage) Ping(_ context.Context) error {
	_, err := os.Stat(s.root)
	return err
}

func (s *DiskStorage) Put(_ context.Context, key string, item PutItem) error {
	filename := s.filename(key)
	if err := os.MkdirAll(filepath.Dir(filename), 0755); err != nil {
		return err
	}

	if item.ContentType != "" {
		if err := ioutil.WriteFile(filename+".mime", []byte(item.ContentType), 0644); err != nil {
			return fmt.Errorf("failed to save item mime-type: %w", err)
		}
	}

	tmpFilename := filename + ".tmp"
	f, err := os.OpenFile(tmpFilename, os.O_RDWR|os.O_CREATE|os.O_TRUNC, 0644)
	if err != nil {
		return err
	}
	defer func() { _ = f.Close() }()

	_, err = io.Copy(f, item.Body)
	if err != nil {
		_ = os.Remove(filename)
		return err
	}

	return os.Rename(tmpFilename, filename)
}

func (s *DiskStorage) Get(_ context.Context, key string) (GetItem, error) {
	filename := s.filename(key)
	fi, err := os.Stat(filename)
	if err != nil {
		return GetItem{}, err
	}

	f, err := os.Open(filename)
	if err != nil {
		return GetItem{}, err
	}

	var ct string
	if fCt, err := ioutil.ReadFile(filename + ".mime"); err == nil {
		ct = string(fCt)
	}

	return GetItem{
		Body:        f,
		UpdatedAt:   fi.ModTime(),
		ContentType: ct,
	}, nil
}

func (s *DiskStorage) Head(_ context.Context, key string) (HeadItem, error) {
	out := HeadItem{}
	filename := s.filename(key)
	fi, err := os.Stat(filename)
	if err != nil {
		return out, err
	}
	out.UpdatedAt = fi.ModTime()

	if fCt, err := ioutil.ReadFile(filename + ".mime"); err == nil {
		out.ContentType = string(fCt)
	}

	return out, nil
}

func (s *DiskStorage) WriteTo(_ context.Context, key string, dst io.Writer) error {
	filename := s.filename(key)
	f, err := os.Open(filename)
	if err != nil {
		return nil
	}
	defer func() { _ = f.Close() }()

	_, err = io.Copy(dst, f)
	return err
}

func (s *DiskStorage) DownloadURI(key string) string {
	return fmt.Sprintf("file://%s", path.Join(filepath.ToSlash(s.root), key))
}

func (s *DiskStorage) filename(key string) string {
	return filepath.Join(s.root, filepath.FromSlash(key))
}
