package storage

import (
	"context"
	"fmt"
	"io"

	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/aws/credentials"
	"github.com/aws/aws-sdk-go/aws/session"
	"github.com/aws/aws-sdk-go/service/s3"
	"github.com/aws/aws-sdk-go/service/s3/s3manager"

	"a.yandex-team.ru/security/kirby/internal/config"
)

const (
	region = "yandex"
)

var _ Storage = (*S3Storage)(nil)

type S3Storage struct {
	cfg      config.S3Storage
	s3       *s3.S3
	uploader *s3manager.Uploader
}

func NewS3Storage(cfg config.S3Storage) (*S3Storage, error) {
	storage := &S3Storage{
		cfg: cfg,
	}

	if err := storage.initS3Client(); err != nil {
		return nil, fmt.Errorf("failed to initialize S3 client: %w", err)
	}

	if err := storage.initBucket(); err != nil {
		return nil, fmt.Errorf("failed to initialize S3 bucket: %w", err)
	}

	storage.initUploader()

	return storage, nil
}

func (s *S3Storage) Ping(ctx context.Context) error {
	_, err := s.s3.HeadBucketWithContext(ctx, &s3.HeadBucketInput{
		Bucket: aws.String(s.cfg.Bucket),
	})
	return err
}

func (s *S3Storage) Put(ctx context.Context, key string, item PutItem) error {
	ctx, cancel := context.WithTimeout(ctx, s.cfg.UploadTimeout)
	defer cancel()

	_, err := s.uploader.UploadWithContext(ctx, &s3manager.UploadInput{
		Bucket:      aws.String(s.cfg.Bucket),
		Key:         aws.String(key),
		Body:        item.Body,
		ContentType: aws.String(item.ContentType),
	})
	return err
}

func (s *S3Storage) Get(ctx context.Context, key string) (GetItem, error) {
	out := GetItem{}
	res, err := s.s3.GetObjectWithContext(ctx, &s3.GetObjectInput{
		Bucket: aws.String(s.cfg.Bucket),
		Key:    aws.String(key),
	})
	if err != nil {
		return out, err
	}

	out.Body = res.Body
	if res.ContentType != nil {
		out.ContentType = *res.ContentType
	}

	if res.LastModified != nil {
		out.UpdatedAt = *res.LastModified
	}

	return out, nil
}

func (s *S3Storage) Head(ctx context.Context, key string) (HeadItem, error) {
	out := HeadItem{}
	res, err := s.s3.HeadObjectWithContext(ctx, &s3.HeadObjectInput{
		Bucket: aws.String(s.cfg.Bucket),
		Key:    aws.String(key),
	})
	if err != nil {
		return out, err
	}

	if res.ContentType != nil {
		out.ContentType = *res.ContentType
	}

	if res.LastModified != nil {
		out.UpdatedAt = *res.LastModified
	}

	return out, nil
}

func (s *S3Storage) WriteTo(ctx context.Context, key string, dst io.Writer) error {
	item, err := s.Get(ctx, key)
	if err != nil {
		return nil
	}
	defer func() { _ = item.Body.Close() }()

	_, err = io.Copy(dst, item.Body)
	return err
}

func (s *S3Storage) DownloadURI(key string) string {
	return fmt.Sprintf("http://%s.%s/%s", s.cfg.Bucket, s.cfg.Endpoint, key)
}

func (s *S3Storage) initBucket() error {
	if _, err := s.s3.HeadBucket(&s3.HeadBucketInput{Bucket: aws.String(s.cfg.Bucket)}); err != nil {
		_, err := s.s3.CreateBucket(&s3.CreateBucketInput{Bucket: aws.String(s.cfg.Bucket)})
		if err != nil {
			return err
		}
	}
	return nil
}

func (s *S3Storage) initS3Client() error {
	s3Credentials := credentials.NewStaticCredentials(
		s.cfg.AccessKeyID,
		s.cfg.AccessSecretKey,
		"",
	)
	_, err := s3Credentials.Get()
	if err != nil {
		err = fmt.Errorf("bad credentials: %w", err)
		return err
	}

	cfg := aws.NewConfig().
		WithRegion(region).
		WithEndpoint(s.cfg.Endpoint).
		WithDisableSSL(true).
		WithCredentials(s3Credentials).
		WithMaxRetries(s.cfg.MaxRetries)

	s3Session, err := session.NewSession()
	if err != nil {
		err = fmt.Errorf("failed to create session: %w", err)
		return err
	}

	s.s3 = s3.New(s3Session, cfg)
	return nil
}

func (s *S3Storage) initUploader() {
	s.uploader = s3manager.NewUploaderWithClient(s.s3)
}
